#!/bin/python
# Tests for file identifiables
# A. Schlemmer, 06/2021

import caosdb as db

import pytest
from pytest import raises

from caoscrawler.identifiable_adapters import LocalStorageIdentifiableAdapter


def test_file_identifiable():
    ident = LocalStorageIdentifiableAdapter()
    file_obj = db.File()

    identifiable = ident.get_identifiable(file_obj)
    identifiable2 = ident.get_identifiable_for_file(file_obj)

    # these are two different objects:
    assert identifiable != identifiable2
    assert file_obj != identifiable
    # ... but the path is equal:
    assert identifiable.path == identifiable2.path
    # ... and very boring:
    assert identifiable.path is None
    # Test functionality of retrieving the files:
    identified_file = ident.get_file(identifiable)
    identified_file2 = ident.get_file(file_obj)
    # The both should be None currently as there are no files in the local store yet:
    assert identified_file is None
    assert identified_file2 is None

    # Let's make it more interesting:
    file_obj.path = "/test/bla/bla.txt"
    file_obj._checksum = "abcd"
    identifiable = ident.get_identifiable(file_obj)
    assert file_obj != identifiable
    assert file_obj.path == identifiable.path
    # Checksum is not part of the identifiable:
    assert file_obj.checksum != identifiable.checksum

    # This is the wrong method, so it should definitely return None:
    identified_file = ident.retrieve_identified_record_for_identifiable(
        identifiable)
    assert identified_file is None
    # This is the correct method to use:
    identified_file = ident.get_file(identifiable)
    # or directly using:
    identified_file2 = ident.get_file(file_obj)
    # The both should be None currently as there are no files in the local store yet:
    assert identified_file is None
    assert identified_file2 is None

    # Try again with actual files in the store:
    records = ident.get_records()
    test_record_wrong_path = db.File(
        path="/bla/bla/test.txt")
    test_record_correct_path = db.File(
        path="/test/bla/bla.txt")
    test_record_alsocorrect_path = db.File(
        path="/test/bla/bla.txt")
    records.append(test_record_wrong_path)
    identified_file = ident.get_file(file_obj)
    assert identified_file is None

    records.append(test_record_correct_path)
    identified_file = ident.get_file(file_obj)
    assert identified_file is not None
    assert identified_file.path == file_obj.path

    with raises(RuntimeError, match=".*unambigiously.*"):
        records.append(test_record_alsocorrect_path)
        identified_file = ident.get_file(file_obj)
