/*
 *
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */

#include "gmock/gmock-matchers.h"             // for ElementsAre, EXPECT_THAT
#include "boost/beast/core/detail/base64.hpp" // for encoded_size
#include "boost/json/object.hpp"              // for object
#include "boost/json/value.hpp"               // for value
#include "caosdb/data_type.h"                 // for atomicdatatype_names
#include "caosdb/entity.h"                    // for importance_names, role...
#include "caosdb/status_code.h"               // for get_status_description
#include "caosdb/utility.h"                   // for base64_encode, load_js...
#include "caosdb_test_utility.h"              // for TEST_DATA_DIR
#include <gtest/gtest-message.h>              // for Message
#include <gtest/gtest-test-part.h>            // for TestPartResult, SuiteA...
#include <gtest/gtest_pred_impl.h>            // for Test, EXPECT_EQ, TestInfo
#include <map>                                // for map, operator!=, _Rb_t...
#include <stdexcept>                          // for out_of_range
#include <string>                             // for allocator, string, ope...
#include <utility>                            // for pair

namespace caosdb::utility {
using ::testing::ElementsAre;

TEST(test_utility, base64_encode) {
  auto test_plain = std::string("admin:caosdb");
  auto test_encoded = std::string("YWRtaW46Y2Fvc2Ri");
  EXPECT_EQ(12, test_plain.size());
  EXPECT_EQ(16, boost::beast::detail::base64::encoded_size(test_plain.size()));
  EXPECT_EQ(test_encoded, base64_encode(test_plain));
}

TEST(test_utility, test_load_json_file) {
  auto json = load_json_file(TEST_DATA_DIR + "/test.json").as_object();

  EXPECT_EQ(json["it"], "tests");
  EXPECT_EQ(json["null values"], nullptr);
  EXPECT_THAT(json["this"].as_array(), ElementsAre("is", "a", "test"));
  EXPECT_THAT(json["numbers"].as_array(), ElementsAre(1, 2, 3.3));
  auto sub = json["arrays and objects"].as_object();
  EXPECT_THAT(sub["see?"].as_array(), ElementsAre(true, false));
}

TEST(test_utility, enum_names) {
  // All working enums
  for (const auto &entry : caosdb::entity::importance_names) {
    EXPECT_EQ(getEnumNameFromValue<caosdb::entity::Importance>(entry.first), entry.second);
    EXPECT_EQ(getEnumValueFromName<caosdb::entity::Importance>(entry.second), entry.first);
  }
  for (const auto &entry : caosdb::entity::role_names) {
    EXPECT_EQ(getEnumNameFromValue<caosdb::entity::Role>(entry.first), entry.second);
    EXPECT_EQ(getEnumValueFromName<caosdb::entity::Role>(entry.second), entry.first);
  }
  for (const auto &entry : caosdb::entity::atomicdatatype_names) {
    EXPECT_EQ(getEnumNameFromValue<caosdb::entity::AtomicDataType>(entry.first), entry.second);
    EXPECT_EQ(getEnumValueFromName<caosdb::entity::AtomicDataType>(entry.second), entry.first);
  }

  // Some non-working examples
  EXPECT_THROW_MESSAGE(getEnumValueFromName<caosdb::entity::Importance>("Invalid name"),
                       std::out_of_range, "Could not find enum value for string 'Invalid name'.");
}

TEST(test_utility, test_status_code_description) {
  EXPECT_EQ(caosdb::get_status_description(12412323), "MISSING DESCRIPTION");
  EXPECT_EQ(caosdb::get_status_description(static_cast<int>(StatusCode::UNKNOWN)),
            "Unknown error. This is typically a bug (server or client).");
}

} // namespace caosdb::utility
