/*
 * This file is a part of the LinkAhead Project.
 *
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021-2024 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */

#ifndef LINKAHEAD_MESSAGE_CODE_H
#define LINKAHEAD_MESSAGE_CODE_H

#include "caosdb/entity/v1/main.pb.h" // for Entity, RepeatedField

/**
 * MessageCodes for entity messages.
 *
 * In contrast to the status codes, the message codes are part of the CaosDB
 * API. Messages (and their codes) represent the state of the entities in a
 * transaction or the server.
 *
 * For a specification of the message codes, look at the protobuf documentation.
 * The sources and documentation can be found at
 * https://gitlab.indiscale.com/caosdb/src/caosdb-proto.
 */

namespace linkahead::entity {

#define _MAP_MESSAGE_CODE(name) name = caosdb::entity::v1::MessageCode::MESSAGE_CODE_##name

enum MessageCode {
  _MAP_MESSAGE_CODE(UNSPECIFIED),
  _MAP_MESSAGE_CODE(UNKNOWN),
  _MAP_MESSAGE_CODE(ENTITY_DOES_NOT_EXIST),
  _MAP_MESSAGE_CODE(ENTITY_HAS_NO_PROPERTIES),
  _MAP_MESSAGE_CODE(ENTITY_HAS_BEEN_DELETED_SUCCESSFULLY),
  _MAP_MESSAGE_CODE(ENTITY_HAS_UNQUALIFIED_PROPERTIES),
  _MAP_MESSAGE_CODE(ENTITY_HAS_UNQUALIFIED_PARENTS),
  _MAP_MESSAGE_CODE(ENTITY_HAS_NO_ID),
  _MAP_MESSAGE_CODE(REQUIRED_BY_PERSISTENT_ENTITY),
  _MAP_MESSAGE_CODE(PROPERTY_HAS_NO_DATA_TYPE),
  _MAP_MESSAGE_CODE(ENTITY_HAS_NO_DESCRIPTION),
  _MAP_MESSAGE_CODE(ENTITY_HAS_NO_NAME),
  _MAP_MESSAGE_CODE(OBLIGATORY_PROPERTY_MISSING),
  _MAP_MESSAGE_CODE(ENTITY_HAS_NO_PARENTS),
  _MAP_MESSAGE_CODE(FILE_HAS_NO_TARGET_PATH),
  _MAP_MESSAGE_CODE(TARGET_PATH_NOT_ALLOWED),
  _MAP_MESSAGE_CODE(TARGET_PATH_EXISTS),
  _MAP_MESSAGE_CODE(PROPERTY_HAS_NO_UNIT),
  _MAP_MESSAGE_CODE(CANNOT_PARSE_VALUE),
  _MAP_MESSAGE_CODE(CHECKSUM_TEST_FAILED),
  _MAP_MESSAGE_CODE(SIZE_TEST_FAILED),
  _MAP_MESSAGE_CODE(CANNOT_CREATE_PARENT_FOLDER),
  _MAP_MESSAGE_CODE(FILE_HAS_NOT_BEEN_UPLOAED),
  _MAP_MESSAGE_CODE(CANNOT_MOVE_FILE_TO_TARGET_PATH),
  _MAP_MESSAGE_CODE(CANNOT_PARSE_DATETIME_VALUE),
  _MAP_MESSAGE_CODE(CANNOT_PARSE_DOUBLE_VALUE),
  _MAP_MESSAGE_CODE(CANNOT_PARSE_INT_VALUE),
  _MAP_MESSAGE_CODE(CANNOT_PARSE_BOOL_VALUE),
  _MAP_MESSAGE_CODE(FILE_NOT_FOUND),
  _MAP_MESSAGE_CODE(WARNING_OCCURED),
  _MAP_MESSAGE_CODE(ENTITY_NAME_IS_NOT_UNIQUE),
  _MAP_MESSAGE_CODE(QUERY_EXCEPTION),
  _MAP_MESSAGE_CODE(TRANSACTION_ROLL_BACK),
  _MAP_MESSAGE_CODE(UNKNOWN_UNIT),
  _MAP_MESSAGE_CODE(AUTHORIZATION_ERROR),
  _MAP_MESSAGE_CODE(REFERENCE_IS_NOT_ALLOWED_BY_DATA_TYPE),
  _MAP_MESSAGE_CODE(ENTITY_NAME_DUPLICATES),
  _MAP_MESSAGE_CODE(DATA_TYPE_NAME_DUPLICATES),
  _MAP_MESSAGE_CODE(ENTITY_HAS_NO_NAME_OR_ID),
  _MAP_MESSAGE_CODE(AFFILIATION_ERROR),
  _MAP_MESSAGE_CODE(QUERY_PARSING_ERROR),
  _MAP_MESSAGE_CODE(NAME_PROPERTIES_MUST_BE_TEXT),
  _MAP_MESSAGE_CODE(PARENT_DUPLICATES_WARNING),
  _MAP_MESSAGE_CODE(PARENT_DUPLICATES_ERROR),
  _MAP_MESSAGE_CODE(ATOMICITY_ERROR),
  _MAP_MESSAGE_CODE(NO_SUCH_ENTITY_ROLE),
  _MAP_MESSAGE_CODE(REQUIRED_BY_UNQUALIFIED),
  _MAP_MESSAGE_CODE(ENTITY_HAS_UNQUALIFIED_REFERENCE),
  _MAP_MESSAGE_CODE(REFERENCED_ENTITY_DOES_NOT_EXIST),
  _MAP_MESSAGE_CODE(REFERENCE_NAME_DUPLICATES),
  _MAP_MESSAGE_CODE(DATA_TYPE_INHERITANCE_AMBIGUOUS),
  _MAP_MESSAGE_CODE(DATA_TYPE_DOES_NOT_ACCEPT_COLLECTION_VALUES),
  _MAP_MESSAGE_CODE(CANNOT_PARSE_UNIT),
  _MAP_MESSAGE_CODE(ADDITIONAL_PROPERTY),
  _MAP_MESSAGE_CODE(PROPERTY_WITH_DATA_TYPE_OVERRIDE),
  _MAP_MESSAGE_CODE(PROPERTY_WITH_DESCRIPTION_OVERRIDE),
  _MAP_MESSAGE_CODE(PROPERTY_WITH_NAME_OVERRIDE),
  _MAP_MESSAGE_CODE(INTEGER_VALUE_OUT_OF_RANGE),
  _MAP_MESSAGE_CODE(INTEGRITY_VIOLATION),
};

[[nodiscard]] inline auto get_message_code(int code) noexcept -> MessageCode {
  // TODO(tf) smarter, less forgot-it-prone implementation
  static MessageCode all_codes[] = {
    MessageCode::UNSPECIFIED,
    MessageCode::UNKNOWN,
    MessageCode::ENTITY_DOES_NOT_EXIST,
    MessageCode::ENTITY_HAS_NO_PROPERTIES,
    MessageCode::ENTITY_HAS_BEEN_DELETED_SUCCESSFULLY,
    MessageCode::ENTITY_HAS_UNQUALIFIED_PROPERTIES,
    MessageCode::ENTITY_HAS_UNQUALIFIED_PARENTS,
    MessageCode::ENTITY_HAS_NO_ID,
    MessageCode::REQUIRED_BY_PERSISTENT_ENTITY,
    MessageCode::PROPERTY_HAS_NO_DATA_TYPE,
    MessageCode::ENTITY_HAS_NO_DESCRIPTION,
    MessageCode::ENTITY_HAS_NO_NAME,
    MessageCode::OBLIGATORY_PROPERTY_MISSING,
    MessageCode::ENTITY_HAS_NO_PARENTS,
    MessageCode::FILE_HAS_NO_TARGET_PATH,
    MessageCode::TARGET_PATH_NOT_ALLOWED,
    MessageCode::TARGET_PATH_EXISTS,
    MessageCode::PROPERTY_HAS_NO_UNIT,
    MessageCode::CANNOT_PARSE_VALUE,
    MessageCode::CHECKSUM_TEST_FAILED,
    MessageCode::SIZE_TEST_FAILED,
    MessageCode::CANNOT_CREATE_PARENT_FOLDER,
    MessageCode::FILE_HAS_NOT_BEEN_UPLOAED,
    MessageCode::CANNOT_MOVE_FILE_TO_TARGET_PATH,
    MessageCode::CANNOT_PARSE_DATETIME_VALUE,
    MessageCode::CANNOT_PARSE_DOUBLE_VALUE,
    MessageCode::CANNOT_PARSE_INT_VALUE,
    MessageCode::CANNOT_PARSE_BOOL_VALUE,
    MessageCode::FILE_NOT_FOUND,
    MessageCode::WARNING_OCCURED,
    MessageCode::ENTITY_NAME_IS_NOT_UNIQUE,
    MessageCode::QUERY_EXCEPTION,
    MessageCode::TRANSACTION_ROLL_BACK,
    MessageCode::UNKNOWN_UNIT,
    MessageCode::AUTHORIZATION_ERROR,
    MessageCode::REFERENCE_IS_NOT_ALLOWED_BY_DATA_TYPE,
    MessageCode::ENTITY_NAME_DUPLICATES,
    MessageCode::DATA_TYPE_NAME_DUPLICATES,
    MessageCode::ENTITY_HAS_NO_NAME_OR_ID,
    MessageCode::AFFILIATION_ERROR,
    MessageCode::QUERY_PARSING_ERROR,
    MessageCode::NAME_PROPERTIES_MUST_BE_TEXT,
    MessageCode::PARENT_DUPLICATES_WARNING,
    MessageCode::PARENT_DUPLICATES_ERROR,
    MessageCode::ATOMICITY_ERROR,
    MessageCode::NO_SUCH_ENTITY_ROLE,
    MessageCode::REQUIRED_BY_UNQUALIFIED,
    MessageCode::ENTITY_HAS_UNQUALIFIED_REFERENCE,
    MessageCode::REFERENCED_ENTITY_DOES_NOT_EXIST,
    MessageCode::REFERENCE_NAME_DUPLICATES,
    MessageCode::DATA_TYPE_INHERITANCE_AMBIGUOUS,
    MessageCode::DATA_TYPE_DOES_NOT_ACCEPT_COLLECTION_VALUES,
    MessageCode::CANNOT_PARSE_UNIT,
    MessageCode::ADDITIONAL_PROPERTY,
    MessageCode::PROPERTY_WITH_DATA_TYPE_OVERRIDE,
    MessageCode::PROPERTY_WITH_DESCRIPTION_OVERRIDE,
    MessageCode::PROPERTY_WITH_NAME_OVERRIDE,
    MessageCode::INTEGER_VALUE_OUT_OF_RANGE,
    MessageCode::INTEGRITY_VIOLATION,
  };

  for (MessageCode known_code : all_codes) {
    if (static_cast<int>(known_code) == code) {
      return known_code;
    }
  }
  return MessageCode::UNKNOWN;
}

} // namespace linkahead::entity
#endif
