/*
 * This file is a part of the CaosDB Project.
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021 Florian Spreckelsen <f.spreckelsen@indiscale.com>
 * Copyright (C) 2021 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public
 * License along with this program. If not, see
 * <https://www.gnu.org/licenses/>.
 *
 */

#ifndef CCAOSDB_H
#define CCAOSDB_H
#ifdef __cplusplus
#include <cstdint> // for int64_t
extern "C" {
#else
#include <stdint.h>  // for int64_t
#include <stdbool.h> // for bool
#endif

/**
 * Return the constant caosdb::LIBCAOSDB_VERSION_MAJOR.
 */
int caosdb_constants_LIBCAOSDB_VERSION_MAJOR();
/**
 * Return the constant caosdb::LIBCAOSDB_VERSION_MINOR
 */
int caosdb_constants_LIBCAOSDB_VERSION_MINOR();
/**
 * Return the constant caosdb::LIBCAOSDB_VERSION_PATCH.
 */
int caosdb_constants_LIBCAOSDB_VERSION_PATCH();
/**
 * Return the constant caosdb::COMPATIBLE_SERVER_VERSION_MAJOR.
 */
int caosdb_constants_COMPATIBLE_SERVER_VERSION_MAJOR();
/**
 * Return the constant caosdb::COMPATIBLE_SERVER_VERSION_MINOR.
 */
int caosdb_constants_COMPATIBLE_SERVER_VERSION_MINOR();
/**
 * Return the constant caosdb::COMPATIBLE_SERVER_VERSION_PATCH.
 */
int caosdb_constants_COMPATIBLE_SERVER_VERSION_PATCH();
/**
 * Return the constant caosdb::COMPATIBLE_SERVER_VERSION_PRE_RELEASE.
 */
const char *caosdb_constants_COMPATIBLE_SERVER_VERSION_PRE_RELEASE();

/**
 * Return the status code reserved for errors in clients wrapping this
 * interface.
 */
int caosdb_status_code_OTHER_CLIENT_ERROR();

/**
 * A wrapper of the C++ Connection class.
 *
 * We use a wrapper for future extensibility and in order to have a minimal
 * capability for type checking in C even though the C++ class
 * Connection is opaque in C.
 */
typedef struct caosdb_connection_connection {
  void *wrapped_connection;
  bool _deletable;
} caosdb_connection_connection;

/**
 * A wrapper of the C++ ConnectionConfiguration class.
 *
 * We use a wrapper for future extensibility and in order to have a minimal
 * capability for type checking in C even though the C++ class
 * Connection is opaque in C.
 */
typedef struct caosdb_connection_connection_configuration {
  void *wrapped_connection_configuration;
  bool _deletable;
} caosdb_connection_connection_configuration;

/**
 * A wrapper of the C++ VersionInfo class.
 *
 * We use a wrapper for future extensibility and in order to have a minimal
 * capability for type checking in C even though the C++ class
 * Connection is opaque in C.
 */
typedef struct caosdb_info_version_info {
  int major;
  int minor;
  int patch;
  const char *pre_release;
  const char *build;
} caosdb_info_version_info;

typedef struct caosdb_connection_certificate_provider {
  void *wrapped_certificate_provider;
  bool _deletable;
} caosdb_connection_certificate_provider;

typedef struct caosdb_authentication_authenticator {
  void *wrapped_authenticator;
  bool _deletable;
} caosdb_authentication_authenticator;

/**
 * Return the environment variable of the given name.
 *
 * If the environment variable is not set, return the fallback instead.
 */
const char *caosdb_utility_get_env_fallback(const char *name, const char *fallback);

/**
 * Return a description of the status code.
 */
const char *caosdb_get_status_description(int code);

/**
 * Create a pem-file certificate provider.
 *
 * Use the destructor function
 * `caosdb_connection_delete_certificate_provider` to free the wrapped
 * provider.
 *
 * EXPERT USE ONLY.  Memory management with this function is a bit tricky.
 * Only use it when you know what you are doing.
 */
int caosdb_connection_create_pem_file_certificate_provider(
  caosdb_connection_certificate_provider *out, const char *path);

/**
 * Destructor function for a certificate provider.
 *
 * EXPERT USE ONLY.  Only use it when you know what you are doing.
 */
int caosdb_connection_delete_certificate_provider(caosdb_connection_certificate_provider *provider);

/**
 * Create a tls-secured connection configuration.
 *
 * The configuration is needed to instantiate a connection.
 *
 * Use the destructor function
 * `caosdb_connection_delete_connection_configuration` to free the wrapped
 * configuration.
 *
 * EXPERT USE ONLY.  Memory management with this function is a bit tricky.
 * Only use it when you know what you are doing.
 */
int caosdb_connection_create_tls_connection_configuration(
  caosdb_connection_connection_configuration *out, const char *host, const int port,
  caosdb_authentication_authenticator *authenticator,
  caosdb_connection_certificate_provider *provider);

/**
 * Create a tls-secured connection configuration.
 *
 * The configuration is needed to instantiate a connection.
 *
 * Use `caosdb_connection_create_tls_connection_configuration` for a
 * tls-secured connection which also supports authentication.
 *
 * Use the destructor function
 * `caosdb_connection_delete_connection_configuration` to free the wrapped
 * configuration.
 *
 * EXPERT USE ONLY.  Memory management with this function is a bit tricky.
 * Only use it when you know what you are doing.
 */
int caosdb_connection_create_insecure_connection_configuration(
  caosdb_connection_connection_configuration *out, const char *host, const int port);

/**
 * Destructor function for the caosdb_connection_connection_configuration
 * struct.
 *
 * EXPERT USE ONLY.  Only use it when you know what you are doing.
 */
int caosdb_connection_delete_connection_configuration(
  caosdb_connection_connection_configuration *configuration);

/**
 * Add a public certificate of a trusted certificate authority to an
 * existing, tls-enabled connection configuration.
 *
 * @param cacert path to a pem-file.
 */
int caosdb_connection_configuration_add_cacert(
  caosdb_connection_connection_configuration *configuration, const char *cacert);

/**
 * Create a plain password authenticator.
 *
 * Use the destructor function
 * `caosdb_authentication_delete_authenticator` to free the wrapped
 * authenticator.
 *
 * EXPERT USE ONLY.  Memory management with this function is a bit tricky.
 * Only use it when you know what you are doing.
 */
int caosdb_authentication_create_plain_password_authenticator(
  caosdb_authentication_authenticator *out, const char *username, const char *password);

/**
 * Destructor function for the caosdb_authentication_authenticator struct.
 *
 * EXPERT USE ONLY.  Only use it when you know what you are doing.
 */
int caosdb_authentication_delete_authenticator(caosdb_authentication_authenticator *authenticator);

/**
 * Create a connection instance.
 *
 * The connection is needed to create transactions and to initiate any other
 * interaction with a CaosDB server.
 *
 * Use the destructor function
 * `caosdb_connection_delete_connection` to free the wrapped
 * connection.
 *
 * EXPERT USE ONLY.  Memory management with this function is a bit tricky.
 * Only use it when you know what you are doing.
 */
int caosdb_connection_create_connection(
  caosdb_connection_connection *out,
  const caosdb_connection_connection_configuration *configuration);

/**
 * Destructor function for the caosdb_connection_connection struct.
 *
 * EXPERT USE ONLY.  Only use it when you know what you are doing.
 */
int caosdb_connection_delete_connection(caosdb_connection_connection *connection);

/**
 * Request the version of the server.
 */
int caosdb_connection_get_version_info(caosdb_info_version_info *out,
                                       const caosdb_connection_connection *connection);

/**
 * Get the default connection from the ConnectionManager.
 *
 * The default connection is to be specified in a configuration file.
 */
int caosdb_connection_connection_manager_get_default_connection(caosdb_connection_connection *out);

/**
 * Get a named connection from the ConnectionManager.
 *
 * The named connection is to be specified in a configuration file.
 */
int caosdb_connection_connection_manager_get_connection(caosdb_connection_connection *out,
                                                        const char *name);

/****************************************************************************
 * ENTITY STUFF AND TRANSACTIONS
 ****************************************************************************/

typedef struct caosdb_transaction_transaction {
  void *wrapped_transaction;
  bool _deletable;
} caosdb_transaction_transaction;

/**
 * Create a transaction on an existing connection.
 *
 * This transaction has to be deleted manually by
 * caosdb_transaction_delete_transaction() later on.
 */
int caosdb_connection_connection_create_transaction(caosdb_connection_connection *connection,
                                                    caosdb_transaction_transaction *out);
int caosdb_transaction_delete_transaction(caosdb_transaction_transaction *transaction);
int caosdb_transaction_transaction_retrieve_by_id(caosdb_transaction_transaction *transaction,
                                                  const char *id);
int caosdb_transaction_transaction_retrieve_and_download_file_by_id(
  caosdb_transaction_transaction *transaction, const char *id, const char *path);
int caosdb_transaction_transaction_retrieve_by_ids(caosdb_transaction_transaction *transaction,
                                                   const char *ids[], int length);
int caosdb_transaction_transaction_query(caosdb_transaction_transaction *transaction,
                                         const char *query);
int caosdb_transaction_transaction_execute(caosdb_transaction_transaction *transaction);
// TODO(fspreck) execute_asynchronously may be added as a separate
// function once we actually support asynchronous execution.

typedef struct caosdb_transaction_result_set {
  void *wrapped_result_set;
  bool _deletable;
} caosdb_transaction_result_set;

typedef struct caosdb_entity_entity {
  void *wrapped_entity;
  bool _deletable;
} caosdb_entity_entity;

int caosdb_transaction_transaction_get_result_set(caosdb_transaction_transaction *transaction,
                                                  caosdb_transaction_result_set *out);
/**
 * Release the result set from the transaction.
 *
 * The transactions is spoiled after this action and should not be used anymore.
 *
 * Note: The result_set has to be deleted via caosdb_transaction_delete_result_set.
 *
 * EXPERT USE ONLY.  Only use it when you know what you are doing.
 */
int caosdb_transaction_transaction_release_result_set(caosdb_transaction_transaction *transaction,
                                                      caosdb_transaction_result_set *out);
/**
 * Release the entity from the result set.
 *
 * Each entity (each index) can be released once. The result set is spoiled
 * after this action and should not be used for anything else anymore.
 *
 * Note: The result_set has to be deleted via caosdb_entity_delete_entity.
 *
 * EXPERT USE ONLY.  Only use it when you know what you are doing.
 */
int caosdb_transaction_result_set_release_at(caosdb_transaction_result_set *result_set,
                                             caosdb_entity_entity *entity, int index);
/**
 * Destructor for caosdb_transaction_result_set.
 *
 * EXPERT USE ONLY.  Only use it when you know what you are doing.
 */
int caosdb_transaction_delete_result_set(caosdb_transaction_result_set *result_set);

int caosdb_transaction_transaction_get_count_result(caosdb_transaction_transaction *transaction,
                                                    long *out);

int caosdb_transaction_result_set_at(caosdb_transaction_result_set *result_set,
                                     caosdb_entity_entity *entity, int index);
int caosdb_transaction_result_set_size(caosdb_transaction_result_set *result_set, int *out);

int caosdb_transaction_transaction_insert_entity(caosdb_transaction_transaction *transaction,
                                                 caosdb_entity_entity *entity);
int caosdb_transaction_transaction_update_entity(caosdb_transaction_transaction *transaction,
                                                 caosdb_entity_entity *entity);
int caosdb_transaction_transaction_delete_by_id(caosdb_transaction_transaction *transaction,
                                                const char *id);

typedef struct caosdb_entity_property {
  void *wrapped_property;
  bool _deletable;
} caosdb_entity_property;

typedef struct caosdb_entity_parent {
  void *wrapped_parent;
  bool _deletable;
} caosdb_entity_parent;

typedef struct caosdb_entity_message {
  void *wrapped_message;
  bool _deletable;
} caosdb_entity_message;

typedef struct caosdb_entity_value {
  void *wrapped_value;
  bool _deletable;
} caosdb_entity_value;

typedef struct caosdb_entity_datatype {
  void *wrapped_datatype;
  bool _deletable;
} caosdb_entity_datatype;

// GETTERS FOR EVERYTHING
int caosdb_entity_entity_get_id(caosdb_entity_entity *entity, char **out);
int caosdb_entity_entity_get_role(caosdb_entity_entity *entity, char **out);
int caosdb_entity_entity_get_name(caosdb_entity_entity *entity, char **out);
int caosdb_entity_entity_get_description(caosdb_entity_entity *entity, char **out);
int caosdb_entity_entity_get_local_path(caosdb_entity_entity *entity, char **out);
int caosdb_entity_entity_get_datatype(caosdb_entity_entity *entity, caosdb_entity_datatype *out);
int caosdb_entity_entity_get_unit(caosdb_entity_entity *entity, char **out);
int caosdb_entity_entity_get_value(caosdb_entity_entity *entity, caosdb_entity_value *out);
int caosdb_entity_entity_get_version_id(caosdb_entity_entity *entity, char **out);
int caosdb_entity_entity_get_errors_size(caosdb_entity_entity *entity, int *out);
int caosdb_entity_entity_get_error(caosdb_entity_entity *entity, caosdb_entity_message *out,
                                   int index);
int caosdb_entity_entity_get_warnings_size(caosdb_entity_entity *entity, int *out);
int caosdb_entity_entity_get_warning(caosdb_entity_entity *entity, caosdb_entity_message *out,
                                     int index);
int caosdb_entity_entity_get_infos_size(caosdb_entity_entity *entity, int *out);
int caosdb_entity_entity_get_info(caosdb_entity_entity *entity, caosdb_entity_message *out,
                                  int index);
int caosdb_entity_entity_get_properties_size(caosdb_entity_entity *entity, int *out);
int caosdb_entity_entity_get_property(caosdb_entity_entity *entity, caosdb_entity_property *out,
                                      int index);
int caosdb_entity_entity_get_parents_size(caosdb_entity_entity *entity, int *out);
int caosdb_entity_entity_get_parent(caosdb_entity_entity *entity, caosdb_entity_parent *out,
                                    int index);

int caosdb_entity_property_get_id(caosdb_entity_property *property, char **out);
int caosdb_entity_property_get_name(caosdb_entity_property *property, char **out);
int caosdb_entity_property_get_description(caosdb_entity_property *property, char **out);
int caosdb_entity_property_get_importance(caosdb_entity_property *property, char **out);
int caosdb_entity_property_get_datatype(caosdb_entity_property *property,
                                        caosdb_entity_datatype *out);
int caosdb_entity_property_get_unit(caosdb_entity_property *property, char **out);
int caosdb_entity_property_get_value(caosdb_entity_property *property, caosdb_entity_value *out);
int caosdb_entity_parent_get_id(caosdb_entity_parent *parent, char **out);
int caosdb_entity_parent_get_name(caosdb_entity_parent *parent, char **out);
int caosdb_entity_parent_get_description(caosdb_entity_parent *parent, char **out);

int caosdb_entity_message_get_code(caosdb_entity_message *message, int *out);
int caosdb_entity_message_get_description(caosdb_entity_message *message, char **out);

int caosdb_entity_datatype_is_undefined(caosdb_entity_datatype *datatype, bool *out);
int caosdb_entity_datatype_is_atomic(caosdb_entity_datatype *datatype, bool *out);
int caosdb_entity_datatype_is_reference(caosdb_entity_datatype *datatype, bool *out);
int caosdb_entity_datatype_is_list_of_atomic(caosdb_entity_datatype *datatype, bool *out);
int caosdb_entity_datatype_is_list_of_reference(caosdb_entity_datatype *datatype, bool *out);
int caosdb_entity_datatype_get_datatype_name(caosdb_entity_datatype *datatype, char **out);

int caosdb_entity_value_is_null(caosdb_entity_value *value, bool *out);
int caosdb_entity_value_is_string(caosdb_entity_value *value, bool *out);
int caosdb_entity_value_is_double(caosdb_entity_value *value, bool *out);
int caosdb_entity_value_is_integer(caosdb_entity_value *value, bool *out);
int caosdb_entity_value_is_bool(caosdb_entity_value *value, bool *out);
int caosdb_entity_value_is_vector(caosdb_entity_value *value, bool *out);
int caosdb_entity_value_get_as_string(caosdb_entity_value *value, char **out);
int caosdb_entity_value_get_as_double(caosdb_entity_value *value, double *out);
int caosdb_entity_value_get_as_integer(caosdb_entity_value *value, int64_t *out);
int caosdb_entity_value_get_as_bool(caosdb_entity_value *value, bool *out);
int caosdb_entity_value_get_as_vector_size(caosdb_entity_value *value, int *out);
int caosdb_entity_value_get_as_vector_at(caosdb_entity_value *value, caosdb_entity_value *out,
                                         const int index);

// CONSTRUCTORS AND DESTRUCTORS
int caosdb_entity_create_entity(caosdb_entity_entity *out);
int caosdb_entity_delete_entity(caosdb_entity_entity *out);
int caosdb_entity_create_property(caosdb_entity_property *out);
int caosdb_entity_delete_property(caosdb_entity_property *out);
int caosdb_entity_create_parent(caosdb_entity_parent *out);
int caosdb_entity_delete_parent(caosdb_entity_parent *out);

// DATATYPE CONSTRUCTORS for atomic and reference datatypes and lists thereof
int caosdb_entity_create_atomic_datatype(caosdb_entity_datatype *out, const char *name);
int caosdb_entity_create_reference_datatype(caosdb_entity_datatype *out, const char *name);
int caosdb_entity_create_atomic_list_datatype(caosdb_entity_datatype *out, const char *name);
int caosdb_entity_create_reference_list_datatype(caosdb_entity_datatype *out, const char *name);
int caosdb_entity_delete_datatype(caosdb_entity_datatype *out);

// VALUE CONSTRUCTORS (resolve overloaded constructors)
int caosdb_entity_create_int_value(caosdb_entity_value *out, const int64_t value);
int caosdb_entity_create_string_value(caosdb_entity_value *out, const char *value);
int caosdb_entity_create_double_value(caosdb_entity_value *out, const double value);
int caosdb_entity_create_bool_value(caosdb_entity_value *out, const bool value);
int caosdb_entity_create_int_vector_value(caosdb_entity_value *out, const int64_t *value,
                                          const int length);
int caosdb_entity_create_string_vector_value(caosdb_entity_value *out, const char **value,
                                             const int length);
int caosdb_entity_create_double_vector_value(caosdb_entity_value *out, const double *value,
                                             const int length);
int caosdb_entity_create_bool_vector_value(caosdb_entity_value *out, const bool *value,
                                           const int length);
int caosdb_entity_delete_value(caosdb_entity_value *out);

// SETTERS FOR EVERYTHING THAT MAY BE SET
int caosdb_entity_entity_set_role(caosdb_entity_entity *entity, const char *role);
int caosdb_entity_entity_set_name(caosdb_entity_entity *entity, const char *name);
int caosdb_entity_entity_set_description(caosdb_entity_entity *entity, const char *description);
int caosdb_entity_entity_set_local_path(caosdb_entity_entity *entity, const char *name);
int caosdb_entity_entity_set_file_path(caosdb_entity_entity *entity, const char *name);
int caosdb_entity_entity_set_datatype(caosdb_entity_entity *entity,
                                      caosdb_entity_datatype *datatype);
int caosdb_entity_entity_set_unit(caosdb_entity_entity *entity, const char *unit);
int caosdb_entity_entity_set_value(caosdb_entity_entity *entity, caosdb_entity_value *value);

int caosdb_entity_entity_append_parent(caosdb_entity_entity *entity, caosdb_entity_parent *parent);
int caosdb_entity_entity_remove_parent(caosdb_entity_entity *entity, int index);
int caosdb_entity_entity_append_property(caosdb_entity_entity *entity,
                                         caosdb_entity_property *property);
int caosdb_entity_entity_remove_property(caosdb_entity_entity *entity, int index);

int caosdb_entity_property_set_id(caosdb_entity_property *property, const char *id);
int caosdb_entity_property_set_name(caosdb_entity_property *property, const char *name);
int caosdb_entity_property_set_datatype(caosdb_entity_property *property,
                                        caosdb_entity_datatype *datatype);
int caosdb_entity_property_set_importance(caosdb_entity_property *property, const char *importance);
int caosdb_entity_property_set_unit(caosdb_entity_property *property, const char *unit);
int caosdb_entity_property_set_value(caosdb_entity_property *property, caosdb_entity_value *value);

int caosdb_entity_parent_set_id(caosdb_entity_parent *parent, const char *id);
int caosdb_entity_parent_set_name(caosdb_entity_parent *parent, const char *name);

#ifdef __cplusplus
}
#endif
#endif
