/*
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */

#ifndef CAOSDB_AUTHENTICATION_H
#define CAOSDB_AUTHENTICATION_H
/**
 * @file caosdb/authentication.h
 * @author Timm Fitschen
 * @date 2021-06-28
 * @brief Configuration and setup of the client authentication.
 */
#include "caosdb/utility.h"               // for base64_encode
#include <grpcpp/security/auth_context.h> // for AuthContext
#include <grpcpp/security/credentials.h>  // for CallCredentials
#include <grpcpp/support/interceptor.h>   // for Status
#include <grpcpp/support/status.h>        // for Status
#include <grpcpp/support/string_ref.h>    // for string_ref
#include <map>                            // for multimap
#include <memory>                         // for shared_ptr
#include <string>                         // for string

namespace caosdb {
namespace authentication {
using caosdb::utility::base64_encode;
using grpc::AuthContext;
using grpc::MetadataCredentialsPlugin;
using grpc::Status;
using grpc::string_ref;

/**
 * @brief Abstract base class for authenticators.
 */
class Authenticator {
public:
  virtual ~Authenticator() = default;
  [[nodiscard]] virtual auto GetCallCredentials() const
    -> std::shared_ptr<grpc::CallCredentials> = 0;
};

/**
 * @brief Implementation of a MetadataCredentialsPlugin which my be created by
 * any implementation of the Authenticator class. Only intended for internal
 * use.
 */
class MetadataCredentialsPluginImpl : public MetadataCredentialsPlugin {
private:
  std::string key;
  std::string value;

public:
  MetadataCredentialsPluginImpl(std::string key, std::string value);

  auto GetMetadata(string_ref service_url, string_ref method_name,
                   const AuthContext &channel_auth_context,
                   std::multimap<grpc::string, grpc::string> *metadata) -> Status override;
};

class PlainPasswordAuthenticator : public Authenticator {
private:
  std::string basic;

public:
  PlainPasswordAuthenticator(const std::string &username, const std::string &password);

  [[nodiscard]] auto GetCallCredentials() const -> std::shared_ptr<grpc::CallCredentials> override;
};
} // namespace authentication
} // namespace caosdb
#endif
