/*
 *
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */
#include "caosdb/configuration.h"  // for ConfigurationError, Configuration...
#include "caosdb/exceptions.h"     // for ConfigurationError
#include "caosdb/log_level.h"      // for CAOSDB_DEFAULT_LOG_LEVEL, CAOSDB_...
#include "caosdb/logging.h"        // for ConsoleSinkConfiguration, Logging...
#include "caosdb_test_utility.h"   // for EXPECT_THROW_MESSAGE, TEST_DATA_DIR
#include <gtest/gtest-message.h>   // for Message
#include <gtest/gtest-test-part.h> // for TestPartResult, SuiteApiResolver
#include <gtest/gtest_pred_impl.h> // for TestInfo, TEST_F, Test
#include <memory>                  // for make_shared
#include <string>                  // for operator+, allocator, string

namespace caosdb::configuration {

class test_configuration : public ::testing::Test {
protected:
  void SetUp() override { ConfigurationManager::Clear(); }
  void TearDown() override { ConfigurationManager::Clear(); }
};

TEST_F(test_configuration, load_json) {
  ConfigurationManager::LoadSingleJSONConfiguration(TEST_DATA_DIR + "/test_caosdb_client.json");
  EXPECT_THROW_MESSAGE(ConfigurationManager::LoadSingleJSONConfiguration("anything"),
                       ConfigurationError, "This CaosDB client has already been configured.");

  ConfigurationManager::Clear();
  EXPECT_THROW_MESSAGE(ConfigurationManager::LoadSingleJSONConfiguration("anything"),
                       ConfigurationError, "Configuration file does not exist.");
  ConfigurationManager::Clear();
}

TEST_F(test_configuration, get_default_connection_configuration_error) {
  EXPECT_THROW_MESSAGE(ConfigurationManager::GetDefaultConnectionName(), ConfigurationError,
                       "This CaosDB client has not been configured.");

  ConfigurationManager::LoadSingleJSONConfiguration(
    TEST_DATA_DIR + "/test_broken_caosdb_client_no_connections1.json");
  EXPECT_THROW_MESSAGE(ConfigurationManager::GetDefaultConnectionName(), ConfigurationError,
                       "This CaosDB client hasn't any configured connections.");
  ConfigurationManager::Clear();

  ConfigurationManager::LoadSingleJSONConfiguration(
    TEST_DATA_DIR + "/test_broken_caosdb_client_no_connections2.json");
  EXPECT_THROW_MESSAGE(ConfigurationManager::GetDefaultConnectionName(), ConfigurationError,
                       "This CaosDB client hasn't any configured connections.");
  ConfigurationManager::Clear();

  ConfigurationManager::LoadSingleJSONConfiguration(
    TEST_DATA_DIR + "/test_broken_caosdb_client_no_connections3.json");
  EXPECT_THROW_MESSAGE(ConfigurationManager::GetDefaultConnectionName(), ConfigurationError,
                       "This CaosDB client hasn't any configured connections.");
  ConfigurationManager::Clear();
}

TEST_F(test_configuration, initialize_logging) {
  auto logging_configuration = caosdb::logging::LoggingConfiguration(CAOSDB_LOG_LEVEL_ALL);
  auto console_sink = std::make_shared<caosdb::logging::ConsoleSinkConfiguration>(
    "console", CAOSDB_DEFAULT_LOG_LEVEL);
  logging_configuration.AddSink(console_sink);

  initialize_logging(logging_configuration);
}

} // namespace caosdb::configuration
