#ifdef __cplusplus
extern "C" {
#endif

/**
 * Return the constant caosdb::LIBCAOSDB_VERSION_MAJOR.
 */
int caosdb_constants_LIBCAOSDB_VERSION_MAJOR();
/**
 * Return the constant caosdb::LIBCAOSDB_VERSION_MINOR
 */
int caosdb_constants_LIBCAOSDB_VERSION_MINOR();
/**
 * Return the constant caosdb::LIBCAOSDB_VERSION_PATCH.
 */
int caosdb_constants_LIBCAOSDB_VERSION_PATCH();
/**
 * Return the constant caosdb::COMPATIBLE_SERVER_VERSION_MAJOR.
 */
int caosdb_constants_COMPATIBLE_SERVER_VERSION_MAJOR();
/**
 * Return the constant caosdb::COMPATIBLE_SERVER_VERSION_MINOR.
 */
int caosdb_constants_COMPATIBLE_SERVER_VERSION_MINOR();
/**
 * Return the constant caosdb::COMPATIBLE_SERVER_VERSION_PATCH.
 */
int caosdb_constants_COMPATIBLE_SERVER_VERSION_PATCH();
/**
 * Return the constant caosdb::COMPATIBLE_SERVER_VERSION_PRE_RELEASE.
 */
const char *caosdb_constants_COMPATIBLE_SERVER_VERSION_PRE_RELEASE();

/**
 * A wrapper of the C++ Connection class.
 *
 * We use a wrapper for future extensibility and in order to have a minimal
 * capability for type checking in C even though the C++ class
 * Connection is opaque in C.
 */
typedef struct {
  void *wrapped_connection;
} caosdb_connection_connection;

/**
 * A wrapper of the C++ ConnectionConfiguration class.
 *
 * We use a wrapper for future extensibility and in order to have a minimal
 * capability for type checking in C even though the C++ class
 * Connection is opaque in C.
 */
typedef struct {
  void *wrapped_connection_configuration;
} caosdb_connection_connection_configuration;

/**
 * A wrapper of the C++ VersionInfo class.
 *
 * We use a wrapper for future extensibility and in order to have a minimal
 * capability for type checking in C even though the C++ class
 * Connection is opaque in C.
 */
typedef struct {
  int major;
  int minor;
  int patch;
  const char *pre_release;
  const char *build;
} caosdb_info_version_info;

typedef struct {
  void *wrapped_certificate_provider;
} caosdb_connection_certificate_provider;

typedef struct {
  void *wrapped_authenticator;
} caosdb_authentication_authenticator;

/**
 * Return the environment variable of the given name.
 *
 * If the environment variable is not set, return the fallback instead.
 */
const char *caosdb_utility_get_env_var(const char *name, const char *fallback);

/**
 * Return a description of the status code.
 */
const char *caosdb_get_status_description(int code);

/**
 * Create a pem-file certificate provider.
 *
 * Use the destructor function
 * `caosdb_connection_delete_certificate_provider` to free the wrapped
 * provider.
 *
 * EXPERT USE ONLY.  Memory management with this function is a bit tricky.
 * Only use it when you know what you are doing.
 */
int caosdb_connection_create_pem_file_certificate_provider(
  caosdb_connection_certificate_provider *out, const char *path);

/**
 * Destructor function for a certificate provider.
 *
 * EXPERT USE ONLY.  Only use it when you know what you are doing.
 */
int caosdb_connection_delete_certificate_provider(
  caosdb_connection_certificate_provider *provider);

/**
 * Create a tls-secured connection configuration.
 *
 * The configuration is needed to instantiate a connection.
 *
 * Use the destructor function
 * `caosdb_connection_delete_connection_configuration` to free the wrapped
 * configuration.
 *
 * EXPERT USE ONLY.  Memory management with this function is a bit tricky.
 * Only use it when you know what you are doing.
 */
int caosdb_connection_create_tls_connection_configuration(
  caosdb_connection_connection_configuration *out, const char *host,
  const int port, caosdb_authentication_authenticator *authenticator,
  caosdb_connection_certificate_provider *provider);

/**
 * Create a tls-secured connection configuration.
 *
 * The configuration is needed to instantiate a connection.
 *
 * Use `caosdb_connection_create_tls_connection_configuration` for a
 * tls-secured connection which also supports authentication.
 *
 * Use the destructor function
 * `caosdb_connection_delete_connection_configuration` to free the wrapped
 * configuration.
 *
 * EXPERT USE ONLY.  Memory management with this function is a bit tricky.
 * Only use it when you know what you are doing.
 */
int caosdb_connection_create_insecure_connection_configuration(
  caosdb_connection_connection_configuration *out, const char *host,
  const int port);

/**
 * Destructor function for the caosdb_connection_connection_configuration
 * struct.
 *
 * EXPERT USE ONLY.  Only use it when you know what you are doing.
 */
int caosdb_connection_delete_connection_configuration(
  caosdb_connection_connection_configuration *configuration);

/**
 * Add a public certificate of a trusted certificate authority to an
 * existing, tls-enabled connection configuration.
 *
 * @param cacert path to a pem-file.
 */
int caosdb_connection_configuration_add_cacert(
  caosdb_connection_connection_configuration *configuration,
  const char *cacert);

/**
 * Create a plain password authenticator.
 *
 * Use the destructor function
 * `caosdb_authentication_delete_authenticator` to free the wrapped
 * authenticator.
 *
 * EXPERT USE ONLY.  Memory management with this function is a bit tricky.
 * Only use it when you know what you are doing.
 */
int caosdb_authentication_create_plain_password_authenticator(
  caosdb_authentication_authenticator *out, const char *username,
  const char *password);

/**
 * Destructor function for the caosdb_authentication_authenticator struct.
 *
 * EXPERT USE ONLY.  Only use it when you know what you are doing.
 */
int caosdb_authentication_delete_authenticator(
  caosdb_authentication_authenticator *authenticator);

/**
 * Create a connection instance.
 *
 * The connection is needed to create transactions and to initiate any other
 * interaction with a CaosDB server.
 *
 * Use the destructor function
 * `caosdb_connection_delete_connection` to free the wrapped
 * connection.
 *
 * EXPERT USE ONLY.  Memory management with this function is a bit tricky.
 * Only use it when you know what you are doing.
 */
int caosdb_connection_create_connection(
  caosdb_connection_connection *out,
  const caosdb_connection_connection_configuration *configuration);

/**
 * Destructor function for the caosdb_connection_connection struct.
 *
 * EXPERT USE ONLY.  Only use it when you know what you are doing.
 */
int caosdb_connection_delete_connection(
  caosdb_connection_connection *connection);

/**
 * Request the version of the server.
 */
int caosdb_connection_get_version_info(
  caosdb_info_version_info *out,
  const caosdb_connection_connection *connection);

/**
 * Get the default connection from the ConnectionManager.
 *
 * The default connection is to be specified in a configuration file.
 */
int caosdb_connection_connection_manager_get_default_connection(
  caosdb_connection_connection *out);

/**
 * Get a named connection from the ConnectionManager.
 *
 * The named connection is to be specified in a configuration file.
 */
int caosdb_connection_connection_manager_get_connection(
  caosdb_connection_connection *out, const char *name);

/****************************************************************************
 * ENTITY STUFF AND TRANSACTIONS
 ****************************************************************************/

// TODO(fspreck) implementations needed, and probably these declarations are
// not sufficient yet.
typedef struct {
  void *wrapped_transaction;
} caosdb_transaction_transaction;

int caosdb_connection_connection_create_transaction(
  caosdb_connection_connection *connection,
  caosdb_transaction_transaction *out);
int caosdb_transaction_transaction_retrieve_by_id(
  caosdb_transaction_transaction *transaction, const char *id);
int caosdb_transaction_transaction_execute(
  caosdb_transaction_transaction *transaction);

typedef struct {
  void *wrapped_result_set;
} caosdb_transaction_result_set;

int caosdb_transaction_transaction_get_result_set(
  caosdb_transaction_transaction *transaction,
  caosdb_transaction_result_set *out);

typedef struct {
  void *wrapped_entity;
  char **id;
  char **role;
  char **name;
  char **description;
  char **datatype;
  char **unit;
  char **value;
  char **version_id;
} caosdb_entity_entity;

int caosdb_transaction_result_set_get_entity(
  caosdb_transaction_result_set *result_set, caosdb_entity_entity *entity,
  int index);

typedef struct {
  void *wrapped_property;
  char **id;
  char **name;
  char **description;
  char **datatype;
  char **unit;
  char **value;
} caosdb_entity_property;
typedef struct {
  void *wrapped_parent;
  char **id;
  char **name;
  char **description;
} caosdb_entity_parent;
typedef struct {
  void *wrapped_message;
  int *code;
  char **description;
} caosdb_entity_message;

// GETTERS FOR COMPLEX OBJECTS
int caosdb_entity_entity_get_errors_size(caosdb_entity_entity *entity,
                                         int *out);
int caosdb_entity_entity_get_error(caosdb_entity_entity *entity,
                                   caosdb_entity_message *out, int index);
int caosdb_entity_entity_get_warnings_size(caosdb_entity_entity *entity,
                                           int *out);
int caosdb_entity_entity_get_warning(caosdb_entity_entity *entity,
                                     caosdb_entity_message *out, int index);
int caosdb_entity_entity_get_infos_size(caosdb_entity_entity *entity, int *out);
int caosdb_entity_entity_get_info(caosdb_entity_entity *entity,
                                  caosdb_entity_message *out, int index);
int caosdb_entity_entity_get_properties_size(caosdb_entity_entity *entity,
                                             int *out);
int caosdb_entity_entity_get_property(caosdb_entity_entity *entity,
                                      caosdb_entity_property *out, int index);
int caosdb_entity_entity_get_parents_size(caosdb_entity_entity *entity,
                                          int *out);
int caosdb_entity_entity_get_parent(caosdb_entity_entity *entity,
                                    caosdb_entity_parent *out, int index);

// CONSTRUCTORS AND DESTRUCTORS
int caosdb_entity_create_entity(caosdb_entity_entity *out);
int caosdb_entity_delete_entity(caosdb_entity_entity *out);
int caosdb_entity_create_property(caosdb_entity_property *out);
int caosdb_entity_delete_property(caosdb_entity_property *out);
int caosdb_entity_create_parent(caosdb_entity_parent *out);
int caosdb_entity_delete_parent(caosdb_entity_parent *out);

// SETTERS FOR COMPLEX OBJECTS
int caosdb_entity_entity_append_parent(caosdb_entity_entity *entity,
                                       caosdb_entity_parent *parent);
int caosdb_entity_entity_append_property(caosdb_entity_entity *entity,
                                         caosdb_entity_property *property);

#ifdef __cplusplus
}
#endif
