/*
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */

#ifndef CAOSDB_UTILS_H
#define CAOSDB_UTILS_H
#include <boost/beast/core/detail/base64.hpp>
#include <boost/filesystem.hpp>
#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/string_file.hpp>
#include <boost/json.hpp>
#include <cassert>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <memory>
#include <string>
#include <string_view>

namespace caosdb::utility {
using boost::filesystem::exists;
using boost::filesystem::ifstream;
using boost::filesystem::path;
using boost::json::stream_parser;
using boost::json::value;

/**
 * @brief Read a text file into a string and return the file's content.
 */
inline auto load_string_file(const path &path) -> std::string {
  std::string result;
  boost::filesystem::load_string_file(path, result);
  return result;
}

/**
 * @brief Return the environment variable KEY, or FALLBACK if it does not exist.
 */
inline auto get_env_var(const char *key, const char *fallback) -> const char * {
  const char *val = getenv(key);
  if (val == nullptr) {
    return fallback;
  } else {
    return val;
  }
}

/**
 * @brief Return the value of an environment variable or - if undefined - the
 * fallback value.
 */
inline auto get_env_var(const std::string &key, const std::string &fallback)
  -> const std::string {
  const char *val = get_env_var(key.c_str(), fallback.c_str());

  auto const result = std::string(val);
  return result;
}

/**
 * @brief Encode string as base64
 */
inline auto base64_encode(const std::string &plain) -> std::string {
  auto size_plain = plain.size();
  auto size_encoded = boost::beast::detail::base64::encoded_size(size_plain);

  std::unique_ptr<char[]> encoded(new char[size_encoded]);
  boost::beast::detail::base64::encode(encoded.get(), plain.c_str(),
                                       size_plain);

  // the encoded char[] is not null terminated, so explicitely set the length
  return std::string(encoded.get(), encoded.get() + size_encoded);
}

inline auto load_json_file(const path &json_file) -> value {
  assert(exists(json_file));

  constexpr auto buffer_size = std::size_t(4096);
  auto stream = ifstream(json_file);
  stream.exceptions(std::ios_base::badbit);

  stream_parser parser;
  auto result = std::string();
  auto buffer = std::string(buffer_size, '\0');
  while (stream.read(&buffer[0], buffer_size)) {
    parser.write(buffer.c_str(), stream.gcount());
  }
  parser.write(buffer.c_str(), stream.gcount());

  assert(parser.done());
  return parser.release();
}

inline auto get_home_directory() -> const path {
  const auto *const home = getenv("HOME");
  // TODO(tf) Add windowsy way of determining the home directory
  return home;
}

} // namespace caosdb::utility
#endif
