/*
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */

#ifndef CAOSDB_MESSAGE_CODE_H
#define CAOSDB_MESSAGE_CODE_H

#include "caosdb/entity/v1alpha1/main.pb.h" // for Entity, RepeatedField

/**
 * MessageCodes for entity messages.
 *
 * In contrast to the status codes, the message codes are part of the CaosDB
 * API. Messages (and their codes) represent the state of the entities in a
 * transaction or the server.
 *
 * For a specification of the message codes, look at the protobuf documentation.
 * The sources and documentation can be found at
 * https://gitlab.indiscale.com/caosdb/src/caosdb-proto.
 */

namespace caosdb::entity {

enum MessageCode {
  UNSPECIFIED = caosdb::entity::v1alpha1::MessageCode::MESSAGE_CODE_UNSPECIFIED,
  UNKNOWN = caosdb::entity::v1alpha1::MessageCode::MESSAGE_CODE_UNKNOWN,
  ENTITY_DOES_NOT_EXIST =
    caosdb::entity::v1alpha1::MessageCode::MESSAGE_CODE_ENTITY_DOES_NOT_EXIST,
  ENTITY_HAS_NO_PROPERTIES = caosdb::entity::v1alpha1::MessageCode::
    MESSAGE_CODE_ENTITY_HAS_NO_PROPERTIES,
};

[[nodiscard]] inline auto get_message_code(int code) noexcept -> MessageCode {
  // TODO(tf) smarter, less forgot-it-prone implementation
  static MessageCode all_codes[] = {
    MessageCode::UNSPECIFIED, MessageCode::UNKNOWN,
    MessageCode::ENTITY_DOES_NOT_EXIST, MessageCode::ENTITY_HAS_NO_PROPERTIES};

  for (MessageCode known_code : all_codes) {
    if (known_code == code) {
      return known_code;
    }
  }
  return MessageCode::UNKNOWN;
}

} // namespace caosdb::entity
#endif
