
/*
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */

#ifndef CAOSDB_LOGGING_H
#define CAOSDB_LOGGING_H

#include "caosdb/log_level.h"                            // for CAOSDB_LOG_...
#include "boost/log/sources/global_logger_storage.hpp"   // for BOOST_LOG_I...
#include "boost/log/sources/record_ostream.hpp"          // IWYU pragma: keep
#include "boost/log/sources/severity_channel_logger.hpp" // for BOOST_LOG_C...
#include "boost/log/utility/setup/settings.hpp"          // for settings
#include "boost/smart_ptr/intrusive_ptr.hpp"             // for intrusive_ptr
#include "boost/smart_ptr/intrusive_ref_counter.hpp"     // for intrusive_p...
#include <memory>                                        // for shared_ptr
#include <string>                                        // for string
#include <vector>                                        // for vector

namespace caosdb::logging {

const std::string logger_name = "caosdb::logging";

typedef boost::log::sources::severity_channel_logger<int, std::string>
  boost_logger_class;

BOOST_LOG_INLINE_GLOBAL_LOGGER_DEFAULT(logger, boost_logger_class)

/**
 * This class stores the integer log level.
 */
class LevelConfiguration {
public:
  LevelConfiguration(int level) : level(level){};
  [[nodiscard]] inline auto GetLevel() const -> int { return this->level; }

private:
  int level;
};

class SinkConfiguration;

/**
 * This class stores the logging level and log sinks.
 *
 * Sinks are represented by SinkConfiguration objects.
 */
class LoggingConfiguration : public LevelConfiguration {
public:
  virtual ~LoggingConfiguration() = default;
  LoggingConfiguration(int level);
  auto AddSink(const std::shared_ptr<SinkConfiguration> &sink) -> void;
  auto GetSinks() const
    -> const std::vector<std::shared_ptr<SinkConfiguration>> &;

private:
  std::vector<std::shared_ptr<SinkConfiguration>> sinks;
};

auto initialize_logging_defaults() -> int;
auto initialize_logging(const LoggingConfiguration &configuration) -> void;

/**
 * A logging sink is characterized by a name and destination.
 *
 * Typical inheriting configurations exist for console, files and syslog.
 *
 * When a SinkConfiguration is created from a configuration, the sink
 * configuration must contain a \p destination key which matches one of the
 * keywords for implemented sinks.  At the moment of writing this documentation,
 * valid destinations are:
 *
 * \li \p file
 * \li \p console
 * \li \p syslog
 *
 * A \p level keyword sets the logging level, if it exists at the sink or
 * logging level of the configuration.
 */
class SinkConfiguration : public LevelConfiguration {
public:
  virtual ~SinkConfiguration() = default;
  SinkConfiguration(std::string name, int level);
  [[nodiscard]] auto GetName() const -> const std::string &;
  [[nodiscard]] virtual auto GetDestination() const -> const std::string & = 0;

  friend auto initialize_logging_defaults() -> int;
  friend auto
  initialize_logging(const LoggingConfiguration &logging_configuration) -> void;

protected:
  virtual auto Configure(boost::log::settings &settings) const -> void;

private:
  std::string name;
};

class ConsoleSinkConfiguration : public SinkConfiguration {
public:
  virtual ~ConsoleSinkConfiguration() = default;
  ConsoleSinkConfiguration(const std::string &name, int level);
  [[nodiscard]] auto GetDestination() const -> const std::string & override;
  friend auto initialize_logging_defaults() -> int;
  friend auto
  initialize_logging(const LoggingConfiguration &logging_configuration) -> void;

protected:
  typedef SinkConfiguration sink_configuration;
  virtual auto Configure(boost::log::settings &settings) const -> void override;

private:
  const std::string destination = "Console";
};

/**
 * The file name is the destination, the directory can be set separately.
 *
 * If there is a `directory` key in the configuration, that will be used as a
 * default, otherwise it is the current directory.
 */
class FileSinkConfiguration : public SinkConfiguration {
public:
  virtual ~FileSinkConfiguration() = default;
  FileSinkConfiguration(const std::string &name, int level);
  [[nodiscard]] virtual auto GetDestination() const
    -> const std::string & override;
  auto SetDirectory(const std::string &directory) -> void;
  friend auto initialize_logging_defaults() -> int;
  friend auto
  initialize_logging(const LoggingConfiguration &logging_configuration) -> void;

protected:
  typedef SinkConfiguration sink_configuration;
  virtual auto Configure(boost::log::settings &settings) const -> void override;

private:
  const std::string destination = "TextFile";
  std::string directory = "./";
};

class SyslogSinkConfiguration : public SinkConfiguration {
public:
  virtual ~SyslogSinkConfiguration() = default;
  SyslogSinkConfiguration(const std::string &name, int level);
  [[nodiscard]] virtual auto GetDestination() const
    -> const std::string & override;

private:
  const std::string destination = "Syslog";
};

/**
 * Convenience function for the C interface.
 */
void caosdb_log_fatal(const char *channel, const char *msg);
/**
 * Convenience function for the C interface.
 */
void caosdb_log_error(const char *channel, const char *msg);
/**
 * Convenience function for the C interface.
 */
void caosdb_log_warn(const char *channel, const char *msg);
/**
 * Convenience function for the C interface.
 */
void caosdb_log_info(const char *channel, const char *msg);
/**
 * Convenience function for the C interface.
 */
void caosdb_log_debug(const char *channel, const char *msg);
/**
 * Convenience function for the C interface.
 */
void caosdb_log_trace(const char *channel, const char *msg);

} // namespace caosdb::logging

#define CAOSDB_LOG_FATAL(Channel)                                              \
  BOOST_LOG_CHANNEL_SEV(caosdb::logging::logger::get(), Channel,               \
                        CAOSDB_LOG_LEVEL_FATAL)
#define CAOSDB_LOG_ERROR(Channel)                                              \
  BOOST_LOG_CHANNEL_SEV(caosdb::logging::logger::get(), Channel,               \
                        CAOSDB_LOG_LEVEL_ERROR)
#define CAOSDB_LOG_WARN(Channel)                                               \
  BOOST_LOG_CHANNEL_SEV(caosdb::logging::logger::get(), Channel,               \
                        CAOSDB_LOG_LEVEL_WARN)
#define CAOSDB_LOG_INFO(Channel)                                               \
  BOOST_LOG_CHANNEL_SEV(caosdb::logging::logger::get(), Channel,               \
                        CAOSDB_LOG_LEVEL_INFO)
#define CAOSDB_LOG_DEBUG(Channel)                                              \
  BOOST_LOG_CHANNEL_SEV(caosdb::logging::logger::get(), Channel,               \
                        CAOSDB_LOG_LEVEL_DEBUG)
#define CAOSDB_LOG_TRACE(Channel)                                              \
  BOOST_LOG_CHANNEL_SEV(caosdb::logging::logger::get(), Channel,               \
                        CAOSDB_LOG_LEVEL_TRACE)

#define CAOSDB_LOG_ERROR_AND_RETURN_STATUS(Channel, StatusCode, Message)       \
  CAOSDB_LOG_ERROR(Channel)                                                    \
    << "StatusCode (" << StatusCode << ") "                                    \
    << caosdb::get_status_description(StatusCode) << ": " << Message;          \
  return StatusCode;

#endif
