/*
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021 Florian Spreckelsen <f.spreckelsen@indiscale.com>
 * Copyright (C) 2021 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */

/**
 * @brief Anything entity-related.
 * @file caosdb/entity.h
 * @author Timm Fitchen
 * @date 2021-07-07
 */
#ifndef CAOSDB_ENTITY_H
#define CAOSDB_ENTITY_H

#include <string>                           // for string
#include "caosdb/entity/v1alpha1/main.pb.h" // for RepeatedPtrField, Message
#include "caosdb/message_code.h"            // for get_message_code, Messag...
#include "google/protobuf/util/json_util.h" // for MessageToJsonString, Jso...

namespace caosdb::entity {
using caosdb::entity::v1alpha1::IdResponse;
using ProtoParent = caosdb::entity::v1alpha1::Parent;
using ProtoProperty = caosdb::entity::v1alpha1::Property;
using ProtoEntity = caosdb::entity::v1alpha1::Entity;

/**
 * Messages convey information about the state and result of transactions.
 *
 * A Message object can be thought of as kinf of a generalized error object in
 * other frameworks. Please have a look at MessageCodes for more details.
 */
class Message {
public:
  [[nodiscard]] inline auto GetCode() const -> MessageCode {
    return get_message_code(wrapped->code());
  }
  [[nodiscard]] inline auto GetDescription() const -> std::string {
    return wrapped->description();
  }

  friend class Entity;
  // TODO(fspreck) Re-enable once we have decided how messages are
  // appended to parents and properties
  // friend class Parent;
  // friend class Property;
  friend class Messages;

private:
  explicit inline Message(caosdb::entity::v1alpha1::Message *wrapped)
    : wrapped(wrapped){};

  caosdb::entity::v1alpha1::Message *wrapped;
};

/**
 * Container for Messages.
 */
class Messages {
public:
  [[nodiscard]] inline auto Size() const -> int { return wrapped->size(); }
  [[nodiscard]] inline auto At(int index) const -> const Message {
    return Message(&(wrapped->at(index)));
  }

  friend class Entity;
  // TODO(fspreck) Same here.
  // friend class Parent;
  // friend class Property;

private:
  inline Messages() : wrapped(nullptr){};

  ::google::protobuf::RepeatedPtrField<caosdb::entity::v1alpha1::Message>
    *wrapped;
};

/**
 * Parent of an Entity.
 *
 * This implementation uses protobuf messages as storage backends. In other
 * words, this class wraps a protobuf message and provides getter and setter
 * methods.
 */
class Parent {
public:
  explicit inline Parent(caosdb::entity::v1alpha1::Parent *wrapped)
    : wrapped(wrapped){};
  Parent();

  /**
   * Return the id of the parent entity.
   */
  [[nodiscard]] auto GetId() const -> const std::string &;
  /**
   * Return the name of the parent entity.
   */
  [[nodiscard]] auto GetName() const -> const std::string &;
  /**
   * Return the description of the parent entity.
   */
  [[nodiscard]] auto GetDescription() const -> const std::string &;

  /**
   * Set the id of the parent.
   */
  auto SetId(const std::string &id) -> void;
  /**
   * Set the name of the parent.
   */
  auto SetName(const std::string &name) -> void;

  /**
   * Return a json string representing this parent.
   *
   * This is intended for debugging.
   */
  inline auto ToString() const -> const std::string {
    google::protobuf::util::JsonOptions options;
    std::string out;
    google::protobuf::util::MessageToJsonString(*(this->wrapped), &out,
                                                options);
    return out;
  }

  // TODO(fspreck) Finish the following implementations once we have
  // decided how to attach messages to parents.
  // /**
  //  * Return the error messages of this parent.
  //  */
  // [[nodiscard]] inline auto GetErrors() const -> const Messages & {
  //   return errors;
  // }
  // [[nodiscard]] inline auto HasErrors() const -> bool {
  //   return this->errors.wrapped->size() > 0;
  // }
  // /**
  //  * Return the warning messages of this parent.
  //  */
  // [[nodiscard]] inline auto GetWarnings() const -> const Messages & {
  //   return warnings;
  // }
  // [[nodiscard]] inline auto HasWarnings() const -> bool {
  //   return this->warnings.wrapped->size() > 0;
  // }
  // /**
  //  * Return the info messages of this parent.
  //  */
  // [[nodiscard]] inline auto GetInfos() const -> const Messages & {
  //   return infos;
  // }

  friend class Entity;
  friend class Parents;

private:
  /**
   * Return an empty protobuf message pointer.
   *
   * This function is called by the default constructor of the
   * caosdb::entity::Parent class and the protobuf message is used as the
   * storage-backend for the new Parent instance.
   *
   * An 'Arena' takes care of the the memory management. Don't try to delete
   * this.
   */
  static auto CreateProtoParent() -> ProtoParent *;

  /**
   * Message which serves as storage backend.
   */
  mutable caosdb::entity::v1alpha1::Parent *wrapped;
  // Messages errors;
  // Messages warnings;
  // Messages infos;
};

/**
 * Container for parents of entities.
 *
 * Should only be instantiated and write-accessed by the owning entity.
 */
class Parents {
public:
  /**
   * Return the current size of the parent container.
   *
   * That is also the number of parents the owning entity currently has.
   */
  [[nodiscard]] inline auto Size() const -> int { return wrapped->size(); }
  /**
   * Return the parent at the given index.
   */
  [[nodiscard]] inline auto At(int index) const -> const Parent {
    return Parent(&(wrapped->at(index)));
  }

  friend class Entity;

private:
  inline Parents(){};
  explicit inline Parents(
    ::google::protobuf::RepeatedPtrField<caosdb::entity::v1alpha1::Parent>
      *wrapped)
    : wrapped(wrapped){};

  /**
   * Append a parent.
   *
   * This increases the Size() by one.
   */
  auto Append(const Parent &parent) -> void;
  /**
   * The collection of parent messages which serves as a backend for this
   * class.
   */
  ::google::protobuf::RepeatedPtrField<caosdb::entity::v1alpha1::Parent>
    *wrapped;
};

/**
 * Property of an Entity.
 *
 * This is a property which belongs to another entity. Don't confuse it with
 * an Entity with the "Property" role.
 */
class Property {
public:
  explicit inline Property(caosdb::entity::v1alpha1::Property *wrapped)
    : wrapped(wrapped){};
  Property();

  /**
   * Return the id of this  property
   */
  [[nodiscard]] auto GetId() const -> const std::string &;
  /**
   * Return the name of this  property
   */
  [[nodiscard]] auto GetName() const -> const std::string &;
  /**
   * Return the description of this  property
   */
  [[nodiscard]] auto GetDescription() const -> const std::string &;
  /**
   * Return the importance of this  property
   */
  [[nodiscard]] auto GetImportance() const -> const std::string &;
  /**
   * Return the value of this  property
   */
  [[nodiscard]] auto GetValue() const -> const std::string &;
  /**
   * Return the unit of this  property
   */
  [[nodiscard]] auto GetUnit() const -> const std::string &;
  /**
   * Return the datatype of this  property
   */
  [[nodiscard]] auto GetDatatype() const -> const std::string &;
  // TODO(fspreck) Implement these when we have decided how to attach
  // messages to properties.
  // [[nodiscard]] auto GetErrors() const -> const Messages &;
  // [[nodiscard]] auto GetWarnings() const -> const Messages &;
  // [[nodiscard]] auto GetInfos() const -> const Messages &;

  /**
   * Set the id of this property.
   */
  auto SetId(const std::string &id) -> void;
  /**
   * Set the name of this property.
   */
  auto SetName(const std::string &name) -> void;
  /**
   * Set the importance of this property.
   */
  auto SetImportance(const std::string &importance) -> void;
  /**
   * Set the value of this property.
   */
  auto SetValue(const std::string &value) -> void;
  /**
   * Set the unit of this property.
   */
  auto SetUnit(const std::string &unit) -> void;
  /**
   * Set the datatype of this property.
   */
  auto SetDatatype(const std::string &datatype) -> void;

  /**
   * Return a json string representing this property.
   *
   * This is intended for debugging
   */
  inline auto ToString() const -> const std::string {
    google::protobuf::util::JsonOptions options;
    std::string out;
    google::protobuf::util::MessageToJsonString(*(this->wrapped), &out,
                                                options);

    return out;
  }

  friend class Entity;
  friend class Properties;

private:
  static auto CreateProtoProperty() -> ProtoProperty *;

  mutable caosdb::entity::v1alpha1::Property *wrapped;
};

/**
 * Container for Properties of Entities.
 *
 * Should only be instantiated and write-accessed by the owning entity.
 */
class Properties {
public:
  /**
   * Return the current size of the properties container.
   *
   * This is also the number of properties the owningn entity currently has.
   */
  [[nodiscard]] inline auto Size() const -> int { return wrapped->size(); }
  /**
   * Return the property at the given index.
   */
  [[nodiscard]] auto At(int index) const -> const Property {
    return Property(&(wrapped->at(index)));
  }

  friend class Entity;

private:
  inline Properties(){};
  explicit inline Properties(
    ::google::protobuf::RepeatedPtrField<caosdb::entity::v1alpha1::Property>
      *wrapped)
    : wrapped(wrapped){};

  /**
   * Append a property
   *
   * This increases the Size() by one.
   */
  auto Append(const Property &property) -> void;

  ::google::protobuf::RepeatedPtrField<caosdb::entity::v1alpha1::Property>
    *wrapped;
};

/**
 * @brief Wrapper for the Protobuf entity.
 */
class Entity {
public:
  Entity();
  inline Entity(const Entity &original) : Entity(CreateProtoEntity()) {
    this->wrapped->CopyFrom(*original.wrapped);
  };
  explicit Entity(IdResponse *idResponse);
  explicit inline Entity(ProtoEntity *wrapped) : wrapped(wrapped) {
    errors.wrapped = this->wrapped->mutable_errors();
    warnings.wrapped = this->wrapped->mutable_warnings();
    infos.wrapped = this->wrapped->mutable_infos();
    properties.wrapped = this->wrapped->mutable_properties();
    parents.wrapped = this->wrapped->mutable_parents();
  };

  [[nodiscard]] inline auto GetId() const noexcept -> const std::string & {
    return wrapped->id();
  };
  [[nodiscard]] inline auto HasId() const noexcept -> bool {
    return !wrapped->id().empty();
  }
  [[nodiscard]] inline auto GetVersionId() const -> const std::string & {
    return wrapped->version().id();
  };

  [[nodiscard]] inline auto GetRole() const -> const std::string & {
    return wrapped->role();
  };
  [[nodiscard]] inline auto GetName() const -> const std::string & {
    return wrapped->name();
  };
  [[nodiscard]] inline auto GetDescription() const -> const std::string & {
    return wrapped->description();
  };

  [[nodiscard]] inline auto GetDatatype() const -> const std::string & {
    return wrapped->datatype();
  };
  [[nodiscard]] inline auto GetUnit() const -> const std::string & {
    return wrapped->unit();
  };
  [[nodiscard]] inline auto GetValue() const -> const std::string & {
    return wrapped->value();
  };

  [[nodiscard]] auto GetParents() const -> const Parents &;
  [[nodiscard]] auto GetProperties() const -> const Properties &;
  [[nodiscard]] inline auto GetErrors() const -> const Messages & {
    return errors;
  }
  [[nodiscard]] inline auto HasErrors() const -> bool {
    return this->errors.wrapped->size() > 0;
  }
  [[nodiscard]] auto GetWarnings() const -> const Messages & {
    return warnings;
  }
  [[nodiscard]] inline auto HasWarnings() const -> bool {
    return this->warnings.wrapped->size() > 0;
  }
  [[nodiscard]] auto GetInfos() const -> const Messages & { return infos; }

  inline auto ToString() const -> const std::string {
    google::protobuf::util::JsonOptions options;
    std::string out;
    google::protobuf::util::MessageToJsonString(*(this->wrapped), &out,
                                                options);
    return out;
  }

  auto SetRole(const std::string &role) -> void;
  auto SetName(const std::string &name) -> void;

  auto SetValue(const std::string &value) -> void;
  auto SetUnit(const std::string &unit) -> void;
  // Currently no references or lists.
  auto SetDatatype(const std::string &datatype) -> void;
  auto AppendProperty(const Property &property) -> void;

  auto AppendParent(const Parent &parent) -> void;
  auto Switch(ProtoEntity *entity) -> void;
  /**
   * Copy all of this entity's features to the target ProtoEntity.
   */
  auto CopyTo(ProtoEntity *target) -> void;

protected:
  static auto CreateProtoEntity() -> ProtoEntity *;
  auto SetId(const std::string &id) -> void;
  auto SetVersionId(const std::string &id) -> void;

private:
  ProtoEntity *wrapped;
  Properties properties;
  Parents parents;
  Messages errors;
  Messages warnings;
  Messages infos;
};

} // namespace caosdb::entity
#endif
