/*
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */

#ifndef CAOSDB_CONFIGURATION_H
#define CAOSDB_CONFIGURATION_H
#include "boost/filesystem/operations.hpp" // for exists
#include "boost/filesystem/path.hpp"       // for path
#include "boost/json/object.hpp"           // for object
#include "boost/json/value.hpp"            // for value
#include "boost/json/value_ref.hpp"        // IWYU pragma: keep
#include "caosdb/authentication.h"         // for Authenticator, PlainPassw...
#include "caosdb/certificate_provider.h"   // for CertificateProvider, path
#include "caosdb/exceptions.h"             // for ConfigurationError
#include "caosdb/logging.h"
#include "caosdb/utility.h"              // for load_json_file
#include "grpcpp/security/credentials.h" // for ChannelCredentials
#include <iosfwd>                        // for ostream
#include <memory>                        // for unique_ptr, shared_ptr
#include <string>                        // for string

namespace caosdb::configuration {
using boost::filesystem::exists;
using boost::filesystem::path;
using boost::json::array;
using boost::json::object;
using boost::json::value;
using caosdb::authentication::Authenticator;
using caosdb::authentication::PlainPasswordAuthenticator;
using caosdb::exceptions::ConfigurationError;
using caosdb::utility::load_json_file;
using grpc::ChannelCredentials;

const std::string logger_name = "caosdb::configuration";

/**
 * @brief Configuration of the CaosDB connection.
 */
class ConnectionConfiguration {
private:
  std::string host;
  int port;

public:
  ConnectionConfiguration(const std::string &host, int port);
  virtual ~ConnectionConfiguration() = default;
  friend auto operator<<(std::ostream &out,
                         const ConnectionConfiguration &configuration)
    -> std::ostream &;

  [[nodiscard]] auto virtual ToString() const -> std::string = 0;
  [[nodiscard]] auto GetHost() const -> std::string;
  [[nodiscard]] auto GetPort() const -> int;
  [[nodiscard]] auto virtual GetChannelCredentials() const
    -> std::shared_ptr<ChannelCredentials> = 0;
};

class InsecureConnectionConfiguration : public ConnectionConfiguration {
private:
  std::shared_ptr<ChannelCredentials> credentials;

public:
  InsecureConnectionConfiguration(const std::string &host, int port);
  [[nodiscard]] auto GetChannelCredentials() const
    -> std::shared_ptr<ChannelCredentials> override;
  [[nodiscard]] auto ToString() const -> std::string override;
};

class TlsConnectionConfiguration : public ConnectionConfiguration {
private:
  std::shared_ptr<ChannelCredentials> credentials;
  std::string certificate_provider;

public:
  TlsConnectionConfiguration(const std::string &host, int port);
  TlsConnectionConfiguration(const std::string &host, int port,
                             const Authenticator &authenticator);
  TlsConnectionConfiguration(const std::string &host, int port,
                             const CertificateProvider &certificate_provider);
  TlsConnectionConfiguration(const std::string &host, int port,
                             const CertificateProvider &certificate_provider,
                             const Authenticator &authenticator);
  [[nodiscard]] auto GetChannelCredentials() const
    -> std::shared_ptr<ChannelCredentials> override;
  [[nodiscard]] auto ToString() const -> std::string override;
};

/**
 * Helper class (no state, just member functions) which should only be used by
 * the ConfigurationManager to initialize the logging framework from the stored
 * configuration.
 */
class LoggingConfigurationHelper {
public:
  friend class ConfigurationManager;

private:
  auto CreateConsoleSinkConfiguration(const object &from,
                                      const std::string &name, int level) const
    -> std::shared_ptr<caosdb::logging::SinkConfiguration>;
  auto CreateSyslogSinkConfiguration(const object &from,
                                     const std::string &name, int level) const
    -> std::shared_ptr<caosdb::logging::SinkConfiguration>;
  auto CreateFileSinkConfiguration(const object &from, const std::string &name,
                                   int level) const
    -> std::shared_ptr<caosdb::logging::SinkConfiguration>;
  auto CreateSinkConfiguration(const object &from, const std::string &name,
                               int default_level) const
    -> std::shared_ptr<caosdb::logging::SinkConfiguration>;
  auto CreateLoggingConfiguration(const object &from) const
    -> caosdb::logging::LoggingConfiguration;
};

/**
 * Helper class (no state, just member functions) which should only be used by
 * the ConfigurationManager to construct Connection instances from the stored
 * configuration.
 */
class ConnectionConfigurationHelper {
public:
  friend class ConfigurationManager;

private:
  /**
   * @param from - a single connection configuration.
   */
  auto CreateCertificateProvider(const object &from) const
    -> std::unique_ptr<CertificateProvider>;

  /**
   * @param from - a single connection configuration.
   */
  auto CreateAuthenticator(const object &from) const
    -> std::unique_ptr<Authenticator>;

  /**
   * @param from - a single connection configuration.
   */
  auto
  CreateConnectionConfiguration(const bool tls, const std::string &host,
                                const int port,
                                const CertificateProvider *certificate_provider,
                                const Authenticator *authenticator) const
    -> std::unique_ptr<ConnectionConfiguration>;

  /**
   * @param from - a single connection configuration.
   */
  auto IsTls(const object &from) const -> bool;

  /**
   * @param from - a single connection configuration.
   */
  auto CreateConnectionConfiguration(const object &from) const
    -> std::unique_ptr<ConnectionConfiguration>;
};

/**
 * Reads the configuration file and keeps the configuration. Singleton.
 *
 * Currently, this class can only read a single configuration file. No merging
 * or overwriting is supported.
 */
class ConfigurationManager {
public:
  static ConfigurationManager &GetInstance() {
    static ConfigurationManager instance;
    return instance;
  };

  /**
   * See mReset.
   */
  inline static auto Reset() noexcept -> int { return GetInstance().mReset(); }

  /**
   * See mClear.
   */
  inline static auto Clear() noexcept -> int { return GetInstance().mClear(); }

  /**
   * See mLoadSingleJSONConfiguration.
   */
  inline static auto LoadSingleJSONConfiguration(const path &json_file)
    -> void {
    GetInstance().mLoadSingleJSONConfiguration(json_file);
  }

  /**
   * See mGetConnectionConfiguration.
   */
  inline static auto GetConnectionConfiguration(const std::string &name)
    -> std::unique_ptr<ConnectionConfiguration> {
    return GetInstance().mGetConnectionConfiguration(name);
  }

  /**
   * Return the ConnectionConfiguration for the default connection.
   */
  inline static auto GetDefaultConnectionConfiguration()
    -> std::unique_ptr<ConnectionConfiguration> {
    return GetInstance().mGetConnectionConfiguration(
      GetInstance().mGetDefaultConnectionName());
  }

  /**
   * See mGetDefaultConnectionName.
   */
  inline static auto GetDefaultConnectionName() -> std::string {
    return GetInstance().mGetDefaultConnectionName();
  }

  ConfigurationManager(ConfigurationManager const &) = delete;
  void operator=(ConfigurationManager const &) = delete;

private:
  value json_configuration;
  ConnectionConfigurationHelper connection_configuration_helper;
  LoggingConfigurationHelper logging_configuration_helper;

  inline ConfigurationManager() { InitializeDefaults(); };

  /**
   * Initialize this ConfigurationManager with the defaults.
   *
   * Currently, this means, that the ConfigurationManager attempts to load the
   * first existing file from the LIBCAOSDB_CONFIGURATION_FILES_PRECEDENCE list
   * of file locations.
   */
  auto InitializeDefaults() -> int;

  /**
   * Return a json object representing the current configuration.
   */
  auto GetConfiguration() const -> const object &;

  /**
   * Return the connection configurations.
   */
  auto GetConnections() const -> const object &;

  /**
   * Return the configuration for the connection with the given name (as a json
   * object).
   */
  auto GetConnection(const std::string &name) const -> const object &;

  /**
   * Reset this ConfigurationManager.
   *
   * The current configuration is deleted and a new configuration is being
   * loaded via InitializeDefaults.
   */
  auto mReset() noexcept -> int;

  /**
   * Clear this ConfigurationManager.
   *
   * Afterwards, this ConfigurationManager is uninitilized.
   *
   * In contrast to mReset, this method only deletes the current configuration
   * but does not load a new one via InitializeDefaults.
   */
  auto mClear() noexcept -> int;

  /**
   * Load a configuration from a json file.
   */
  auto mLoadSingleJSONConfiguration(const path &json_file) -> void;

  /**
   * Return the ConnectionConfiguration for the connection of the given name.
   */
  auto mGetConnectionConfiguration(const std::string &name) const
    -> std::unique_ptr<ConnectionConfiguration>;

  /**
   * Return the ConnectionConfiguration for the default connection.
   */
  auto mGetDefaultConnectionName() const -> std::string;
};

} // namespace caosdb::configuration
#endif
