/*
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2022 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2022 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */
#include "caosdb/acm/user.h"       // for User
#include "caosdb/connection.h"     // for Connection, VersionInfo, path
#include "caosdb/exceptions.h"     // for TransactionError
#include "caosdb_test_utility.h"   // for EXPECT_THROW_MESSAGE
#include <gtest/gtest-message.h>   // for Message
#include <gtest/gtest-test-part.h> // for SuiteApiResolver, TestPartR...
#include <gtest/gtest_pred_impl.h> // for Test, TestInfo, TEST, EXPEC...
#include <memory>                  // for allocator, unique_ptr, __sh...
#include <string>                  // for stoi, string

namespace caosdb::connection {
using caosdb::acm::User;

TEST(test_user, test_create_single_user_failure_invalid_name) {
  auto connection = ConnectionManager::GetDefaultConnection();
  User user("CaosDB", "-user1");
  EXPECT_THROW_STARTS_WITH(
    connection->CreateSingleUser(user), caosdb::exceptions::Exception,
    "The attempt to execute this transaction was not successful because an error occured in the "
    "transport or RPC protocol layer. Original error: 2 - The user name does not comply with the "
    "current policies for user names:");
}

TEST(test_user, test_create_single_user_failure_invalid_password) {
  auto connection = ConnectionManager::GetDefaultConnection();
  User user("CaosDB", "user1");
  user.SetPassword("1234");
  EXPECT_THROW_STARTS_WITH(
    connection->CreateSingleUser(user), caosdb::exceptions::Exception,
    "The attempt to execute this transaction was not successful because an error occured in the "
    "transport or RPC protocol layer. Original error: 2 - The password does not comply with the "
    "current policies for passwords: ");
}

TEST(test_user, test_create_single_user_success) {
  auto connection = ConnectionManager::GetDefaultConnection();
  User user("CaosDB", "user1");
  user.SetPassword("Password1!");
  connection->CreateSingleUser(user);
  auto re = connection->RetrieveSingleUser("CaosDB", "user1");
  EXPECT_EQ(re.GetName(), user.GetName());
  connection->DeleteSingleUser("CaosDB", "user1");
  EXPECT_THROW_MESSAGE(
    connection->RetrieveSingleUser("CaosDB", "user1"), caosdb::exceptions::Exception,
    "The attempt to execute this transaction was not successful because an error occured in the "
    "transport or RPC protocol layer. Original error: 5 - This account does not exist.");
}

} // namespace caosdb::connection
