/*
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */
#include "caosdb/connection.h"         // for Connection, ConnectionManager
#include "caosdb/data_type.h"          // for AtomicDataType
#include "caosdb/entity.h"             // for Entity, Messages, Message
#include "caosdb/transaction.h"        // for Entity, Transaction,...
#include "caosdb/transaction_status.h" // for TransactionStatus, StatusCode
#include "caosdb/value.h"              // for value
#include <cstdint>                     // for int64_t
#include <gtest/gtest-message.h>       // for Message
#include <gtest/gtest-test-part.h>     // for TestPartResult, SuiteApiResolver
#include <gtest/gtest_pred_impl.h>     // for Test, EXPECT_EQ, AssertionResult
#include <iostream>                    // for cout
#include <memory>                      // for unique_ptr, allocator, __shar...
#include <string>                      // for string
#include <vector>                      // for vector

namespace caosdb::entity {

class test_list_properties : public ::testing::Test {
protected:
  void SetUp() override { DeleteEntities(); }

  void TearDown() override { DeleteEntities(); }

  static void DeleteEntities() {
    const auto &connection = caosdb::connection::ConnectionManager::GetDefaultConnection();
    auto query_transaction(connection->CreateTransaction());
    query_transaction->Query("FIND ENTITY WITH id > 99");
    query_transaction->Execute();
    if (query_transaction->GetResultSet().size() > 0) {
      auto delete_transaction(connection->CreateTransaction());
      for (const Entity &entity : query_transaction->GetResultSet()) {
        delete_transaction->DeleteById(entity.GetId());
      }
      delete_transaction->Execute();
    }
  }
};

TEST_F(test_list_properties, insert_list_of_text) {
  const auto &connection = caosdb::connection::ConnectionManager::GetDefaultConnection();

  auto insertion_prop(connection->CreateTransaction());

  Entity abstract_list_property;
  abstract_list_property.SetRole(Role::PROPERTY);
  abstract_list_property.SetName("TestProp");
  abstract_list_property.SetDataType(DataType::ListOf(AtomicDataType::TEXT));
  abstract_list_property.SetValue(std::vector<std::string>{"item1", "item2", "item3"});

  insertion_prop->InsertEntity(&abstract_list_property);
  std::cout << "response " << insertion_prop->ResponseToString();
  insertion_prop->Execute();
  EXPECT_TRUE(insertion_prop->GetStatus().IsTerminated());

  auto insertion_rt(connection->CreateTransaction());

  Property list_property;
  list_property.SetId(insertion_prop->GetResultSet().at(0).GetId());
  list_property.SetValue(std::vector<std::string>{"item4", "item5", "item6"});

  Entity entity;
  entity.SetRole(Role::RECORD_TYPE);
  entity.SetName("TestRT");
  entity.AppendProperty(list_property);

  insertion_rt->InsertEntity(&entity);
  std::cout << "response " << insertion_rt->ResponseToString();
  insertion_rt->Execute();
  EXPECT_TRUE(insertion_rt->GetStatus().IsTerminated());
  EXPECT_FALSE(insertion_rt->GetStatus().IsError());

  // retrieve and check again
  auto retrieval(connection->CreateTransaction());
  retrieval->RetrieveById(insertion_rt->GetResultSet().at(0).GetId());
  retrieval->Execute();

  EXPECT_TRUE(retrieval->GetStatus().IsTerminated());
  EXPECT_FALSE(retrieval->GetStatus().IsError());

  const auto &same_entity = retrieval->GetResultSet().at(0);
  const auto &data_type = same_entity.GetProperties().at(0).GetDataType();
  const auto &value = same_entity.GetProperties().at(0).GetValue();

  EXPECT_TRUE(data_type.IsList());
  EXPECT_TRUE(data_type.GetAsList().IsListOfAtomic());
  EXPECT_EQ(data_type.GetAsList().GetAtomicDataType(), AtomicDataType::TEXT);

  EXPECT_TRUE(value.IsVector());
  EXPECT_EQ(value.GetAsVector().size(), 3);
  EXPECT_TRUE(value.GetAsVector().at(1).IsString());
  EXPECT_EQ(value.GetAsVector().at(1).GetAsString(), "item5");
}

TEST_F(test_list_properties, insert_list_of_int) {
  const auto &connection = caosdb::connection::ConnectionManager::GetDefaultConnection();

  auto insertion_prop(connection->CreateTransaction());

  Entity abstract_list_property;
  abstract_list_property.SetRole(Role::PROPERTY);
  abstract_list_property.SetName("TestProp");
  abstract_list_property.SetDataType(DataType::ListOf(AtomicDataType::INTEGER));
  abstract_list_property.SetValue(std::vector<int64_t>{1, 2, 3});

  insertion_prop->InsertEntity(&abstract_list_property);
  std::cout << "response " << insertion_prop->ResponseToString();
  insertion_prop->Execute();
  EXPECT_TRUE(insertion_prop->GetStatus().IsTerminated());

  auto insertion_rt(connection->CreateTransaction());

  Property list_property;
  list_property.SetId(insertion_prop->GetResultSet().at(0).GetId());
  list_property.SetValue(std::vector<int64_t>{4, 5, 6});

  Entity entity;
  entity.SetRole(Role::RECORD_TYPE);
  entity.SetName("TestRT");
  entity.AppendProperty(list_property);

  insertion_rt->InsertEntity(&entity);
  std::cout << "response " << insertion_rt->ResponseToString();
  insertion_rt->Execute();
  EXPECT_TRUE(insertion_rt->GetStatus().IsTerminated());
  EXPECT_FALSE(insertion_rt->GetStatus().IsError());

  // retrieve and check again
  auto retrieval(connection->CreateTransaction());
  retrieval->RetrieveById(insertion_rt->GetResultSet().at(0).GetId());
  retrieval->Execute();

  EXPECT_TRUE(retrieval->GetStatus().IsTerminated());
  EXPECT_FALSE(retrieval->GetStatus().IsError());

  const auto &same_entity = retrieval->GetResultSet().at(0);
  const auto &data_type = same_entity.GetProperties().at(0).GetDataType();
  const auto &value = same_entity.GetProperties().at(0).GetValue();

  EXPECT_TRUE(data_type.IsList());
  EXPECT_TRUE(data_type.GetAsList().IsListOfAtomic());
  EXPECT_EQ(data_type.GetAsList().GetAtomicDataType(), AtomicDataType::INTEGER);

  EXPECT_TRUE(value.IsVector());
  EXPECT_EQ(value.GetAsVector().size(), 3);
  EXPECT_TRUE(value.GetAsVector().at(1).IsInt64());
  EXPECT_EQ(value.GetAsVector().at(1).GetAsInt64(), 5);
}

TEST_F(test_list_properties, insert_list_of_bool) {
  const auto &connection = caosdb::connection::ConnectionManager::GetDefaultConnection();

  auto insertion_prop(connection->CreateTransaction());

  Entity abstract_list_property;
  abstract_list_property.SetRole(Role::PROPERTY);
  abstract_list_property.SetName("TestProp");
  abstract_list_property.SetDataType(DataType::ListOf(AtomicDataType::BOOLEAN));
  abstract_list_property.SetValue(std::vector<bool>{true, true, false});

  insertion_prop->InsertEntity(&abstract_list_property);
  std::cout << "response " << insertion_prop->ResponseToString();
  insertion_prop->Execute();
  EXPECT_TRUE(insertion_prop->GetStatus().IsTerminated());

  auto insertion_rt(connection->CreateTransaction());

  Property list_property;
  list_property.SetId(insertion_prop->GetResultSet().at(0).GetId());
  list_property.SetValue(std::vector<bool>{false, false, true});

  Entity entity;
  entity.SetRole(Role::RECORD_TYPE);
  entity.SetName("TestRT");
  entity.AppendProperty(list_property);

  insertion_rt->InsertEntity(&entity);
  std::cout << "response " << insertion_rt->ResponseToString();
  insertion_rt->Execute();
  EXPECT_TRUE(insertion_rt->GetStatus().IsTerminated());
  EXPECT_FALSE(insertion_rt->GetStatus().IsError());

  // retrieve and check again
  auto retrieval(connection->CreateTransaction());
  retrieval->RetrieveById(insertion_rt->GetResultSet().at(0).GetId());
  retrieval->Execute();

  EXPECT_TRUE(retrieval->GetStatus().IsTerminated());
  EXPECT_FALSE(retrieval->GetStatus().IsError());

  const auto &same_entity = retrieval->GetResultSet().at(0);
  const auto &data_type = same_entity.GetProperties().at(0).GetDataType();
  const auto &value = same_entity.GetProperties().at(0).GetValue();

  EXPECT_TRUE(data_type.IsList());
  EXPECT_TRUE(data_type.GetAsList().IsListOfAtomic());
  EXPECT_EQ(data_type.GetAsList().GetAtomicDataType(), AtomicDataType::BOOLEAN);

  EXPECT_TRUE(value.IsVector());
  EXPECT_EQ(value.GetAsVector().size(), 3);
  EXPECT_TRUE(value.GetAsVector().at(1).IsBool());
  EXPECT_FALSE(value.GetAsVector().at(1).GetAsBool());
}

TEST_F(test_list_properties, insert_list_non_list_datatype) {
  const auto &connection = caosdb::connection::ConnectionManager::GetDefaultConnection();

  auto insertion_prop(connection->CreateTransaction());

  Entity abstract_list_property;
  abstract_list_property.SetRole(Role::PROPERTY);
  abstract_list_property.SetName("TestProp");
  abstract_list_property.SetDataType(AtomicDataType::TEXT);
  abstract_list_property.SetValue(std::vector<std::string>{"item1", "item2", "item3"});

  insertion_prop->InsertEntity(&abstract_list_property);
  std::cout << "response " << insertion_prop->ResponseToString();
  insertion_prop->ExecuteAsynchronously();
  insertion_prop->WaitForIt();
  EXPECT_TRUE(insertion_prop->GetStatus().IsTerminated());
  EXPECT_TRUE(insertion_prop->GetStatus().IsError());

  auto query_transaction(connection->CreateTransaction());
  query_transaction->Query("FIND ENTITY TestProp");
  query_transaction->Execute();
  EXPECT_EQ(query_transaction->GetResultSet().size(), 0);
}

} // namespace caosdb::entity
