/*
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 */
#include "caosdb/authentication.h"       // for PlainPasswordAuthenticator
#include "caosdb/certificate_provider.h" // for PemFileCertificateProvider
#include "caosdb/configuration.h"        // for InsecureConnectionConfigura...
#include "caosdb/connection.h"           // for Connection, VersionInfo, path
#include "caosdb/constants.h"            // for COMPATIBLE_SERVER_VERSION_M...
#include "caosdb/exceptions.h"           // for AuthenticationError, Connec...
#include "caosdb/info.h"                 // for VersionInfo
#include "caosdb/utility.h"              // for get_env_fallback
#include "caosdb_test_utility.h"         // for EXPECT_THROW_MESSAGE
#include <filesystem>                    // for path
#include <gtest/gtest_pred_impl.h> // NOLINT TODO how to fix this? for Test, TestInfo, TEST, EXPEC...
#include <gtest/gtest-message.h>   // for Message
#include <gtest/gtest-test-part.h> // for SuiteApiResolver, TestPartR...
#include <memory>                  // for allocator, unique_ptr, __sh...
#include <string>                  // for stoi, string

namespace caosdb::connection {
using caosdb::authentication::PlainPasswordAuthenticator;
using caosdb::configuration::InsecureConnectionConfiguration;
using caosdb::configuration::PemFileCertificateProvider;
using caosdb::configuration::TlsConnectionConfiguration;
using caosdb::exceptions::AuthenticationError;
using caosdb::exceptions::ConnectionError;

TEST(test_connection, config_somehost_25323) {
  auto port = 25323;
  const auto *host = "somehost";
  auto config = InsecureConnectionConfiguration(host, port);

  EXPECT_EQ(host, config.GetHost());
  EXPECT_EQ(port, config.GetPort());
}

TEST(test_connection, connect_somehost_42347_fails) {
  auto port = 42347;
  const auto *host = "somehost";
  auto config = InsecureConnectionConfiguration(host, port);
  Connection connection(config);

  EXPECT_THROW(connection.RetrieveVersionInfo(), ConnectionError);
}

TEST(test_connection, connection_insecure_authentication_error_anonymous) {
  const auto *port_str = caosdb::utility::get_env_fallback("CAOSDB_SERVER_GRPC_PORT_HTTP", "8080");
  auto port = std::stoi(port_str);
  const auto *const host = caosdb::utility::get_env_fallback("CAOSDB_SERVER_HOST", "localhost");

  auto config = InsecureConnectionConfiguration(host, port);
  auto connection = Connection(config);

  EXPECT_THROW(connection.RetrieveVersionInfo(), AuthenticationError);
}

TEST(test_connection, connection_ssl_authentication_error_anonymous) {
  const auto *port_str = caosdb::utility::get_env_fallback("CAOSDB_SERVER_GRPC_PORT_HTTPS", "8443");
  auto port = std::stoi(port_str);
  const auto *const host = caosdb::utility::get_env_fallback("CAOSDB_SERVER_HOST", "localhost");
  const auto path = caosdb::utility::get_env_fallback("CAOSDB_SERVER_CERT", std::string());

  auto cert = PemFileCertificateProvider(path);
  auto config = TlsConnectionConfiguration(host, port, cert);
  auto connection = Connection(config);

  EXPECT_THROW_MESSAGE(connection.RetrieveVersionInfo(), AuthenticationError,
                       "The attempt to execute this transaction has not been "
                       "executed at all because the authentication did not "
                       "succeed. Original error: Please log in!");
}

TEST(test_connection, connection_ssl_authentication_error_wrong_credentials) {
  const auto *port_str = caosdb::utility::get_env_fallback("CAOSDB_SERVER_GRPC_PORT_HTTPS", "8443");
  auto port = std::stoi(port_str);
  const auto *const host = caosdb::utility::get_env_fallback("CAOSDB_SERVER_HOST", "localhost");
  const auto path = caosdb::utility::get_env_fallback("CAOSDB_SERVER_CERT", std::string());
  const auto *const user = "not-a-user-23461237";
  const auto *const password = "very-c-cred";

  auto auth = PlainPasswordAuthenticator(user, password);
  auto cert = PemFileCertificateProvider(path);
  auto config = TlsConnectionConfiguration(host, port, cert, auth);
  auto connection = Connection(config);

  EXPECT_THROW_MESSAGE(connection.RetrieveVersionInfo(), AuthenticationError,
                       "The attempt to execute this transaction has not been executed at all "
                       "because the authentication did not succeed. Original error: "
                       "Authentication failed. Username or password wrong.");
}

TEST(test_connection, connection_ssl_authentication_success) {
  const auto &connection = ConnectionManager::GetDefaultConnection();

  auto major = caosdb::COMPATIBLE_SERVER_VERSION_MAJOR;
  auto minor = caosdb::COMPATIBLE_SERVER_VERSION_MINOR;
  auto patch = caosdb::COMPATIBLE_SERVER_VERSION_PATCH;
  const auto pre_release = std::string(caosdb::COMPATIBLE_SERVER_VERSION_PRE_RELEASE);

  const auto &v_info = connection->RetrieveVersionInfo();

  EXPECT_EQ(major, v_info.GetMajor());
  EXPECT_LE(minor, v_info.GetMinor());
  if (minor == v_info.GetMinor()) {
    EXPECT_LE(patch, v_info.GetPatch());
  }
  if (!pre_release.empty()) {
    EXPECT_EQ(pre_release, v_info.GetPreRelease());
  }
}

} // namespace caosdb::connection
