/*
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2021 Timm Fitschen <t.fitschen@indiscale.com>
 * Copyright (C) 2021 Florian Spreckelsen <f.spreckelsen@indiscale.com>
 * Copyright (C) 2021 IndiScale GmbH <info@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */
#include "caosdb/file_transmission/file_writer.h" // for FileWriter
#include "caosdb/message_code.h"                  // for ENTITY_DOES_NOT_EXIST
#include "caosdb/status_code.h"                   // for GO_ON, StatusCode
#include "ccaosdb.h"                              // for caosdb_connection_...
#include <chrono>                                 // for filesystem
#include <cstring>                                // for strcmp
#include <filesystem>                             // for remove
#include <gtest/gtest-message.h>                  // for Message
#include <gtest/gtest-test-part.h>                // for TestPartResult
#include <gtest/gtest_pred_impl.h>                // for EXPECT_EQ, TestInfo
#include <iostream>                               // for operator<<, endl
#include <string>                                 // for allocator, string

namespace fs = std::filesystem;
class test_ccaosdb : public ::testing::Test {
protected:
  fs::path test_upload_file_1;
  fs::path test_download_file_1;

  void SetUp() override {
    DeleteEntities();
    test_upload_file_1 = fs::path("test_upload_file_1_delete_me.dat");
    test_download_file_1 = fs::path("test_download_file_1_delete_me.dat");

    // fill the file that shall be uploaded
    caosdb::transaction::FileWriter writer(test_upload_file_1);
    std::string buffer(1024, 'c');
    for (int i = 0; i < 20; i++) {
      writer.write(buffer);
    }
  }

  void TearDown() override {
    // delete files
    fs::remove(test_upload_file_1);
    fs::remove(test_download_file_1);

    DeleteEntities();
  }

  /**
   * Delete everything with id>99, C style
   */
  static void DeleteEntities() {
    caosdb_connection_connection connection;
    caosdb_connection_connection_manager_get_default_connection(&connection);

    caosdb_transaction_transaction query_transaction;
    caosdb_connection_connection_create_transaction(&connection, &query_transaction);
    caosdb_transaction_transaction_query(&query_transaction, "FIND ENTITY WITH id > 99");
    caosdb_transaction_transaction_execute(&query_transaction);

    caosdb_transaction_result_set result_set;
    caosdb_transaction_transaction_get_result_set(&query_transaction, &result_set);
    int count[] = {-1}; // NOLINT
    caosdb_transaction_result_set_size(&result_set, count);
    if (*count > 0) {
      caosdb_transaction_transaction delete_transaction;
      caosdb_connection_connection_create_transaction(&connection, &delete_transaction);
      caosdb_entity_entity entity;
      char *id = nullptr; // NOLINT
      for (int i = 0; i < *count; i++) {
        caosdb_transaction_result_set_at(&result_set, &entity, i);
        caosdb_entity_entity_get_id(&entity, &id);
        caosdb_transaction_transaction_delete_by_id(&delete_transaction, id);
      }

      caosdb_transaction_transaction_execute(&delete_transaction);
      caosdb_transaction_delete_transaction(&delete_transaction);
    }

    caosdb_transaction_delete_transaction(&query_transaction);
  }
};

TEST_F(test_ccaosdb, connection_ssl_authentication_success) {

  caosdb_connection_connection connection;
  caosdb_connection_connection_manager_get_default_connection(&connection);

  caosdb_info_version_info version_info;
  caosdb_connection_get_version_info(&version_info, &connection);

  auto major = caosdb_constants_COMPATIBLE_SERVER_VERSION_MAJOR();
  auto minor = caosdb_constants_COMPATIBLE_SERVER_VERSION_MINOR();
  auto patch = caosdb_constants_COMPATIBLE_SERVER_VERSION_PATCH();
  const auto *const pre_release = caosdb_constants_COMPATIBLE_SERVER_VERSION_PRE_RELEASE();

  EXPECT_EQ(major, version_info.major);
  EXPECT_LE(minor, version_info.minor);
  if (minor == version_info.minor) {
    EXPECT_LE(patch, version_info.patch);
  }
  if (strcmp(pre_release, "") != 0) {
    EXPECT_STREQ(pre_release, version_info.pre_release);
  }
}

TEST_F(test_ccaosdb, test_count_query) {

  caosdb_connection_connection connection;
  int return_code(caosdb_connection_connection_manager_get_default_connection(&connection));
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_transaction transaction;
  return_code = caosdb_connection_connection_create_transaction(&connection, &transaction);
  EXPECT_EQ(return_code, 0);

  // COUNT query with an empty result
  return_code = caosdb_transaction_transaction_query(&transaction, "COUNT ENTITY WITH id=-1");
  EXPECT_EQ(return_code, caosdb::StatusCode::GO_ON);

  return_code = caosdb_transaction_transaction_execute(&transaction);
  EXPECT_EQ(return_code, 0);

  long dummy(-1);       // NOLINT
  long *count = &dummy; // NOLINT
  return_code = caosdb_transaction_transaction_get_count_result(&transaction, count);
  EXPECT_EQ(return_code, 0);
  EXPECT_EQ(*count, 0);

  return_code = caosdb_transaction_delete_transaction(&transaction);
  EXPECT_EQ(return_code, 0);
}

TEST_F(test_ccaosdb, test_query) {
  caosdb_connection_connection connection;
  int return_code(caosdb_connection_connection_manager_get_default_connection(&connection));
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_transaction transaction;
  return_code = caosdb_connection_connection_create_transaction(&connection, &transaction);
  EXPECT_EQ(return_code, 0);

  return_code = caosdb_transaction_transaction_query(&transaction, "FIND ENTITY WITH id=-1");
  EXPECT_EQ(return_code, caosdb::StatusCode::GO_ON);

  return_code = caosdb_transaction_transaction_execute(&transaction);
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_result_set result_set;
  return_code = caosdb_transaction_transaction_get_result_set(&transaction, &result_set);
  EXPECT_EQ(return_code, 0);

  int dummy(-1);
  int *count = &dummy; // NOLINT
  return_code = caosdb_transaction_result_set_size(&result_set, count);
  EXPECT_EQ(return_code, 0);
  EXPECT_EQ(*count, 0);

  return_code = caosdb_transaction_delete_transaction(&transaction);
  EXPECT_EQ(return_code, 0);
}

TEST_F(test_ccaosdb, test_single_id_retrieve) {
  caosdb_connection_connection connection;
  int return_code(caosdb_connection_connection_manager_get_default_connection(&connection));
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_transaction transaction;
  return_code = caosdb_connection_connection_create_transaction(&connection, &transaction);
  EXPECT_EQ(return_code, 0);

  return_code = caosdb_transaction_transaction_retrieve_by_id(&transaction, "21");
  EXPECT_EQ(return_code, caosdb::StatusCode::GO_ON);

  return_code = caosdb_transaction_transaction_execute(&transaction);
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_result_set result_set;
  return_code = caosdb_transaction_transaction_get_result_set(&transaction, &result_set);
  EXPECT_EQ(return_code, 0);

  int dummy(-1);
  int *count = &dummy; // NOLINT
  return_code = caosdb_transaction_result_set_size(&result_set, count);
  EXPECT_EQ(return_code, 0);
  EXPECT_EQ(*count, 1);

  caosdb_entity_entity entity;
  return_code = caosdb_transaction_result_set_at(&result_set, &entity, 0);
  EXPECT_EQ(return_code, 0);

  return_code = caosdb_entity_entity_get_errors_size(&entity, count);
  EXPECT_EQ(return_code, 0);
  EXPECT_EQ(*count, 0);

  char *out = nullptr; // NOLINT
  return_code = caosdb_entity_entity_get_name(&entity, &out);
  EXPECT_EQ(return_code, 0);
  EXPECT_STREQ(out, "unit");

  return_code = caosdb_transaction_delete_transaction(&transaction);
  EXPECT_EQ(return_code, 0);
}

TEST_F(test_ccaosdb, test_multi_id_retrieve) {
  caosdb_connection_connection connection;
  int return_code(caosdb_connection_connection_manager_get_default_connection(&connection));
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_transaction transaction;
  return_code = caosdb_connection_connection_create_transaction(&connection, &transaction);
  EXPECT_EQ(return_code, 0);

  const char *ids[] = {"20", "21", "22"}; // NOLINT
  return_code = caosdb_transaction_transaction_retrieve_by_ids(&transaction, ids, 3);
  EXPECT_EQ(return_code, caosdb::StatusCode::GO_ON);

  return_code = caosdb_transaction_transaction_execute(&transaction);
  // Should have an error since entity 22 doesn't exist
  EXPECT_TRUE((return_code > 0));

  caosdb_transaction_result_set result_set;
  return_code = caosdb_transaction_transaction_get_result_set(&transaction, &result_set);
  EXPECT_EQ(return_code, 0);

  int dummy(-1);
  int *count = &dummy; // NOLINT
  return_code = caosdb_transaction_result_set_size(&result_set, count);
  EXPECT_EQ(return_code, 0);
  EXPECT_EQ(*count, 3);

  caosdb_entity_entity entity;
  return_code = caosdb_transaction_result_set_at(&result_set, &entity, 1);
  EXPECT_EQ(return_code, 0);

  return_code = caosdb_entity_entity_get_errors_size(&entity, count);
  EXPECT_EQ(return_code, 0);
  EXPECT_EQ(*count, 0);

  char *out = nullptr; // NOLINT
  return_code = caosdb_entity_entity_get_name(&entity, &out);
  EXPECT_EQ(return_code, 0);
  EXPECT_STREQ(out, "unit");

  caosdb_entity_entity other_entity;
  return_code = caosdb_transaction_result_set_at(&result_set, &other_entity, 2);
  EXPECT_EQ(return_code, 0);
  return_code = caosdb_entity_entity_get_errors_size(&other_entity, count);
  EXPECT_EQ(return_code, 0);
  EXPECT_EQ(*count, 1);

  caosdb_entity_message err_msg;
  return_code = caosdb_entity_entity_get_error(&other_entity, &err_msg, 0);
  EXPECT_EQ(return_code, 0);

  return_code = caosdb_entity_message_get_code(&err_msg, count);
  EXPECT_EQ(return_code, 0);
  EXPECT_EQ(*count, caosdb::entity::MessageCode::ENTITY_DOES_NOT_EXIST);

  return_code = caosdb_transaction_delete_transaction(&transaction);
  EXPECT_EQ(return_code, 0);
}

TEST_F(test_ccaosdb, test_insert_update_delete) {

  caosdb_connection_connection connection;
  caosdb_connection_connection_manager_get_default_connection(&connection);

  caosdb_entity_entity original_entity;
  caosdb_entity_create_entity(&original_entity);
  caosdb_entity_entity_set_name(&original_entity, "TestName");
  caosdb_entity_entity_set_role(&original_entity, "PROPERTY");
  caosdb_entity_datatype in_type;
  caosdb_entity_create_atomic_datatype(&in_type, "TEXT");
  caosdb_entity_entity_set_datatype(&original_entity, &in_type);

  caosdb_transaction_transaction insert_transaction;
  caosdb_connection_connection_create_transaction(&connection, &insert_transaction);
  caosdb_transaction_transaction_insert_entity(&insert_transaction, &original_entity);
  int return_code(caosdb_transaction_transaction_execute(&insert_transaction));
  EXPECT_EQ(return_code, 0);

  // Insert the entity and check results
  caosdb_transaction_result_set insert_result_set;
  caosdb_transaction_transaction_get_result_set(&insert_transaction, &insert_result_set);
  int count[] = {-1}; // NOLINT
  caosdb_transaction_result_set_size(&insert_result_set, count);
  EXPECT_EQ(*count, 1);
  caosdb_entity_entity inserted_entity;
  caosdb_transaction_result_set_at(&insert_result_set, &inserted_entity, 0);
  char *in = nullptr; // NOLINT
  caosdb_entity_entity_get_id(&inserted_entity, &in);

  // Retrieve it again for checks and update
  caosdb_transaction_transaction retrieve_transaction_1;
  caosdb_connection_connection_create_transaction(&connection, &retrieve_transaction_1);
  caosdb_transaction_transaction_retrieve_by_id(&retrieve_transaction_1, in);
  return_code = caosdb_transaction_transaction_execute(&retrieve_transaction_1);
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_result_set retrieve_results_1;
  caosdb_transaction_transaction_get_result_set(&retrieve_transaction_1, &retrieve_results_1);
  caosdb_transaction_result_set_size(&retrieve_results_1, count);
  EXPECT_EQ(*count, 1);
  caosdb_entity_entity retrieved_entity_1;
  caosdb_transaction_result_set_at(&retrieve_results_1, &retrieved_entity_1, 0);
  char *out = nullptr; // NOLINT
  caosdb_entity_entity_get_id(&retrieved_entity_1, &out);
  EXPECT_EQ(strcmp(in, out), 0);

  caosdb_entity_entity_get_name(&original_entity, &in);
  caosdb_entity_entity_get_name(&retrieved_entity_1, &out);
  EXPECT_EQ(strcmp(in, out), 0);

  caosdb_entity_entity_get_role(&original_entity, &in);
  caosdb_entity_entity_get_role(&retrieved_entity_1, &out);
  EXPECT_EQ(strcmp(in, out), 0);

  caosdb_entity_datatype out_type;
  caosdb_entity_entity_get_datatype(&original_entity, &out_type);
  bool out_is(false);
  caosdb_entity_datatype_is_atomic(&out_type, &out_is);
  EXPECT_TRUE(out_is);
  caosdb_entity_datatype_get_datatype_name(&in_type, &in);
  caosdb_entity_datatype_get_datatype_name(&out_type, &out);
  EXPECT_EQ(strcmp(in, out), 0);

  // Change name and update
  return_code = caosdb_entity_entity_set_name(&retrieved_entity_1, "TestNameNew");
  EXPECT_EQ(return_code, 0);
  caosdb_transaction_transaction update_transaction;
  caosdb_connection_connection_create_transaction(&connection, &update_transaction);
  caosdb_transaction_transaction_update_entity(&update_transaction, &retrieved_entity_1);
  return_code = caosdb_transaction_transaction_execute(&update_transaction);
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_result_set update_results;
  caosdb_transaction_transaction_get_result_set(&update_transaction, &update_results);
  caosdb_transaction_result_set_size(&update_results, count);
  EXPECT_EQ(*count, 1);
  caosdb_entity_entity updated_entity;
  caosdb_transaction_result_set_at(&update_results, &updated_entity, 0);
  caosdb_entity_entity_get_id(&inserted_entity, &in);
  caosdb_entity_entity_get_id(&updated_entity, &out);
  EXPECT_EQ(strcmp(in, out), 0);

  // Retrieve again
  caosdb_transaction_transaction retrieve_transaction_2;
  caosdb_connection_connection_create_transaction(&connection, &retrieve_transaction_2);
  caosdb_transaction_transaction_retrieve_by_id(&retrieve_transaction_2, out);
  return_code = caosdb_transaction_transaction_execute(&retrieve_transaction_2);
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_result_set retrieve_results_2;
  caosdb_transaction_transaction_get_result_set(&retrieve_transaction_2, &retrieve_results_2);
  caosdb_transaction_result_set_size(&retrieve_results_2, count);
  EXPECT_EQ(*count, 1);
  caosdb_entity_entity retrieved_entity_2;
  caosdb_transaction_result_set_at(&retrieve_results_2, &retrieved_entity_2, 0);

  caosdb_entity_entity_get_id(&retrieved_entity_2, &out);
  EXPECT_EQ(strcmp(in, out), 0);

  caosdb_entity_entity_get_name(&retrieved_entity_1, &in);
  caosdb_entity_entity_get_name(&retrieved_entity_2, &out);
  EXPECT_EQ(strcmp(in, out), 0);

  // Doesn't have the old name anymore
  caosdb_entity_entity_get_name(&original_entity, &in);
  EXPECT_FALSE((strcmp(in, out) == 0));

  // Everything else hasn't changed
  caosdb_entity_entity_get_role(&original_entity, &in);
  caosdb_entity_entity_get_role(&retrieved_entity_2, &out);
  EXPECT_EQ(strcmp(in, out), 0);

  caosdb_entity_entity_get_datatype(&retrieved_entity_2, &out_type);
  caosdb_entity_datatype_is_atomic(&out_type, &out_is);
  EXPECT_TRUE(out_is);
  caosdb_entity_datatype_get_datatype_name(&in_type, &in);
  caosdb_entity_datatype_get_datatype_name(&out_type, &out);
  EXPECT_EQ(strcmp(in, out), 0);

  // Now delete
  caosdb_transaction_transaction delete_transaction;
  caosdb_connection_connection_create_transaction(&connection, &delete_transaction);
  caosdb_entity_entity_get_id(&retrieved_entity_2, &in);
  caosdb_transaction_transaction_delete_by_id(&delete_transaction, in);
  return_code = caosdb_transaction_transaction_execute(&delete_transaction);
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_result_set delete_results;
  caosdb_transaction_transaction_get_result_set(&delete_transaction, &delete_results);
  caosdb_transaction_result_set_size(&delete_results, count);
  EXPECT_EQ(*count, 1);

  caosdb_entity_entity deleted_entity;
  caosdb_transaction_result_set_at(&delete_results, &deleted_entity, 0);
  caosdb_entity_entity_get_id(&deleted_entity, &out);
  EXPECT_EQ(strcmp(in, out), 0);

  // Try to retrieve again
  caosdb_transaction_transaction retrieve_transaction_3;
  caosdb_connection_connection_create_transaction(&connection, &retrieve_transaction_3);
  caosdb_transaction_transaction_retrieve_by_id(&retrieve_transaction_3, out);
  return_code = caosdb_transaction_transaction_execute(&retrieve_transaction_3);
  EXPECT_TRUE((return_code > 0));

  caosdb_transaction_result_set retrieve_results_3;
  caosdb_transaction_transaction_get_result_set(&retrieve_transaction_3, &retrieve_results_3);
  caosdb_transaction_result_set_size(&retrieve_results_3, count);
  EXPECT_EQ(*count, 1);
  caosdb_entity_entity retrieved_entity_3;
  caosdb_transaction_result_set_at(&retrieve_results_3, &retrieved_entity_3, 0);
  caosdb_entity_entity_get_errors_size(&retrieved_entity_3, count);
  EXPECT_EQ(*count, 1);
  caosdb_entity_message err_msg;
  caosdb_entity_entity_get_error(&retrieved_entity_3, &err_msg, 0);
  caosdb_entity_message_get_code(&err_msg, count);
  EXPECT_EQ(*count, caosdb::entity::MessageCode::ENTITY_DOES_NOT_EXIST);

  // Delete everything
  caosdb_transaction_delete_transaction(&retrieve_transaction_3);
  caosdb_transaction_delete_transaction(&delete_transaction);
  caosdb_transaction_delete_transaction(&retrieve_transaction_2);
  caosdb_transaction_delete_transaction(&update_transaction);
  caosdb_transaction_delete_transaction(&retrieve_transaction_1);
  caosdb_transaction_delete_transaction(&insert_transaction);
  caosdb_entity_delete_entity(&original_entity);
  caosdb_entity_delete_datatype(&in_type);
}

TEST_F(test_ccaosdb, test_insert_with_prop_and_parent) {

  caosdb_connection_connection connection;
  caosdb_connection_connection_manager_get_default_connection(&connection);

  // Create and insert a property, ...
  std::cout << "Creating a property..." << std::endl;
  caosdb_entity_entity original_prop;
  caosdb_entity_create_entity(&original_prop);

  caosdb_entity_entity_set_name(&original_prop, "TestProp");
  caosdb_entity_entity_set_role(&original_prop, "PROPERTY");
  caosdb_entity_datatype original_type;
  caosdb_entity_create_atomic_datatype(&original_type, "TEXT");
  caosdb_entity_entity_set_datatype(&original_prop, &original_type);

  std::cout << "Inserting a property..." << std::endl;
  caosdb_transaction_transaction prop_insertion;
  caosdb_connection_connection_create_transaction(&connection, &prop_insertion);
  caosdb_transaction_transaction_insert_entity(&prop_insertion, &original_prop);
  int return_code(caosdb_transaction_transaction_execute(&prop_insertion));
  EXPECT_EQ(return_code, 0);

  std::cout << "Checking result..." << std::endl;
  caosdb_transaction_result_set prop_results;
  caosdb_transaction_transaction_get_result_set(&prop_insertion, &prop_results);
  caosdb_entity_entity inserted_prop;
  caosdb_transaction_result_set_at(&prop_results, &inserted_prop, 0);
  char *prop_id = nullptr; // NOLINT
  caosdb_entity_entity_get_id(&inserted_prop, &prop_id);

  // ... a record type with this property, ...
  caosdb_entity_entity original_rt;
  caosdb_entity_create_entity(&original_rt);

  caosdb_entity_entity_set_name(&original_rt, "TestType");
  caosdb_entity_entity_set_role(&original_rt, "RECORD_TYPE");

  caosdb_entity_property rt_prop;
  caosdb_entity_create_property(&rt_prop);
  caosdb_entity_property_set_id(&rt_prop, prop_id);
  caosdb_entity_entity_append_property(&original_rt, &rt_prop);

  caosdb_transaction_transaction rt_insertion;
  caosdb_connection_connection_create_transaction(&connection, &rt_insertion);
  caosdb_transaction_transaction_insert_entity(&rt_insertion, &original_rt);
  return_code = caosdb_transaction_transaction_execute(&rt_insertion);
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_result_set rt_results;
  caosdb_transaction_transaction_get_result_set(&rt_insertion, &rt_results);
  caosdb_entity_entity inserted_rt;
  caosdb_transaction_result_set_at(&rt_results, &inserted_rt, 0);
  char *rt_id = nullptr; // NOLINT
  caosdb_entity_entity_get_id(&inserted_rt, &rt_id);

  // ... and a record with this record type as a parent.
  caosdb_entity_entity original_rec;
  caosdb_entity_create_entity(&original_rec);
  caosdb_entity_entity_set_name(&original_rec, "TestRec");
  caosdb_entity_entity_set_role(&original_rec, "RECORD");
  caosdb_entity_parent rec_parent;
  caosdb_entity_create_parent(&rec_parent);
  caosdb_entity_parent_set_id(&rec_parent, rt_id);
  caosdb_entity_entity_append_parent(&original_rec, &rec_parent);
  caosdb_entity_property rec_prop;
  caosdb_entity_create_property(&rec_prop);
  caosdb_entity_property_set_id(&rec_prop, prop_id);
  caosdb_entity_value in_value;
  caosdb_entity_create_string_value(&in_value, "Bla");
  caosdb_entity_property_set_value(&rec_prop, &in_value);
  caosdb_entity_entity_append_property(&original_rec, &rec_prop);

  caosdb_transaction_transaction rec_insertion;
  caosdb_connection_connection_create_transaction(&connection, &rec_insertion);
  caosdb_transaction_transaction_insert_entity(&rec_insertion, &original_rec);
  return_code = caosdb_transaction_transaction_execute(&rec_insertion);
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_result_set rec_results;
  caosdb_transaction_transaction_get_result_set(&rec_insertion, &rec_results);
  caosdb_entity_entity inserted_rec;
  caosdb_transaction_result_set_at(&rec_results, &inserted_rec, 0);
  char *rec_id = nullptr; // NOLINT
  caosdb_entity_entity_get_id(&inserted_rec, &rec_id);

  // Retrieve the record again for comparison
  caosdb_transaction_transaction retrieve_transaction;
  caosdb_connection_connection_create_transaction(&connection, &retrieve_transaction);
  caosdb_transaction_transaction_retrieve_by_id(&retrieve_transaction, rec_id);
  return_code = caosdb_transaction_transaction_execute(&retrieve_transaction);
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_result_set retrieve_result;
  caosdb_transaction_transaction_get_result_set(&retrieve_transaction, &retrieve_result);
  caosdb_entity_entity retrieved_rec;
  caosdb_transaction_result_set_at(&retrieve_result, &retrieved_rec, 0);
  char *in = nullptr;  // NOLINT
  char *out = nullptr; // NOLINT
  caosdb_entity_entity_get_name(&original_rec, &in);
  caosdb_entity_entity_get_name(&retrieved_rec, &out);
  EXPECT_EQ(strcmp(in, out), 0);
  int count[] = {0}; // NOLINT
  caosdb_entity_entity_get_properties_size(&retrieved_rec, count);
  EXPECT_EQ(*count, 1);
  caosdb_entity_entity_get_parents_size(&retrieved_rec, count);
  EXPECT_EQ(*count, 1);

  caosdb_entity_parent retrieved_parent;
  caosdb_entity_entity_get_parent(&retrieved_rec, &retrieved_parent, 0);
  caosdb_entity_entity_get_name(&original_rt, &in);
  caosdb_entity_parent_get_name(&retrieved_parent, &out);
  EXPECT_EQ(strcmp(in, out), 0);
  caosdb_entity_parent_get_id(&retrieved_parent, &out);
  EXPECT_EQ(strcmp(rt_id, out), 0);

  caosdb_entity_property retrieved_property;
  caosdb_entity_entity_get_property(&retrieved_rec, &retrieved_property, 0);
  caosdb_entity_entity_get_name(&original_prop, &in);
  caosdb_entity_property_get_name(&retrieved_property, &out);
  EXPECT_EQ(strcmp(in, out), 0);
  caosdb_entity_value out_value;
  caosdb_entity_property_get_value(&retrieved_property, &out_value);
  caosdb_entity_value_get_as_string(&in_value, &in);
  caosdb_entity_value_get_as_string(&out_value, &out);
  EXPECT_EQ(strcmp(in, out), 0);

  caosdb_transaction_delete_transaction(&retrieve_transaction);
  caosdb_transaction_delete_transaction(&rec_insertion);
  caosdb_entity_delete_property(&rec_prop);
  caosdb_entity_delete_parent(&rec_parent);
  caosdb_entity_delete_entity(&original_rec);
  caosdb_transaction_delete_transaction(&rt_insertion);
  caosdb_entity_delete_property(&rt_prop);
  caosdb_entity_delete_entity(&original_rt);
  caosdb_transaction_delete_transaction(&prop_insertion);
  caosdb_entity_delete_entity(&original_prop);
  caosdb_entity_delete_datatype(&original_type);
  caosdb_entity_delete_value(&in_value);
}

TEST_F(test_ccaosdb, test_up_n_download_file) {

  caosdb_connection_connection connection;
  caosdb_connection_connection_manager_get_default_connection(&connection);

  caosdb_entity_entity original_entity;
  caosdb_entity_create_entity(&original_entity);
  caosdb_entity_entity_set_role(&original_entity, "FILE");
  caosdb_entity_entity_set_local_path(&original_entity, test_upload_file_1.c_str());
  caosdb_entity_entity_set_file_path(&original_entity, "/some/file");

  caosdb_transaction_transaction insert_transaction;
  caosdb_connection_connection_create_transaction(&connection, &insert_transaction);
  caosdb_transaction_transaction_insert_entity(&insert_transaction, &original_entity);
  int return_code(caosdb_transaction_transaction_execute(&insert_transaction));
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_result_set rec_results;
  caosdb_transaction_transaction_get_result_set(&insert_transaction, &rec_results);
  caosdb_entity_entity inserted_rec;
  caosdb_transaction_result_set_at(&rec_results, &inserted_rec, 0);
  char *rec_id = nullptr; // NOLINT
  caosdb_entity_entity_get_id(&inserted_rec, &rec_id);

  caosdb_entity_entity download_entity;
  caosdb_entity_create_entity(&download_entity);
  caosdb_entity_entity_set_role(&download_entity, "FILE");
  caosdb_entity_entity_set_local_path(&download_entity, "lol");
  caosdb_entity_entity_set_file_path(&download_entity, "/some/file");

  caosdb_transaction_transaction retrieve_transaction_1;
  caosdb_connection_connection_create_transaction(&connection, &retrieve_transaction_1);
  EXPECT_EQ(return_code, 0);
  // return_code = caosdb_transaction_transaction_retrieve_by_id(
  //   &retrieve_transaction_1, rec_id);
  return_code = caosdb_transaction_transaction_retrieve_and_download_file_by_id(
    &retrieve_transaction_1, rec_id, test_download_file_1.c_str());
  EXPECT_EQ(return_code, caosdb::StatusCode::GO_ON);
  return_code = caosdb_transaction_transaction_execute(&retrieve_transaction_1);
  EXPECT_EQ(return_code, 0);

  caosdb_transaction_result_set result_set;
  return_code = caosdb_transaction_transaction_get_result_set(&retrieve_transaction_1, &result_set);
  EXPECT_EQ(return_code, 0);
}
